/* apccomm_all.c
   Part of "APCComm" 
   Copyright (C) 2000-2025 Ralf Hoffmann
   Contact: ralf@boomerangsworld.de

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include "apccomm_all.h"

#if defined AMIGA
#include "apccomm_am.h"
#include "apccomm_am_main.h"

#if defined GUI
#include "apccomm_gui_includes.h"
#include "apccomm_gui.h"
#endif

#endif

const char *aboutStr = "APComm %ld.%ld.%ld\n" \
                 "Copyright (C) 2000-2025 Ralf Hoffmann\n" \
                 "\n" \
                 "This program is free software; you can redistribute it and/or modify\n" \
                 "it under the terms of the GNU General Public License as published by\n" \
                 "the Free Software Foundation; either version 2 of the License, or\n" \
                 "(at your option) any later version.\n" \
                 "\n" \
                 "This program is distributed in the hope that it will be useful,\n" \
                 "but WITHOUT ANY WARRANTY; without even the implied warranty of\n" \
                 "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n" \
                 "GNU General Public License for more details.\n" \
                 "\n" \
                 "You should have received a copy of the GNU General Public License\n" \
                 "along with this program; if not, write to the Free Software\n" \
                 "Foundation, Inc., 59 Temple Place - Suite 330,\n" \
                 "Boston, MA 02111-1307, USA.\n" \
                 "\n" \
                 "Use this at your own risk. A basic checksum code is used but there is\n" \
                 "no warranty for correct transmission. You have been warned!\n" \
                 "I'm not responsible for any error or damage that occurs by using\n" \
                 "this program.";

const char *aboutStr2 = "For more information or new version visit the homepage:\n" \
                  "http://www.boomerangsworld.de/apccomm\n" \
                  "\n" \
                  "E-Mail: Ralf Hoffmann <ralf@boomerangsworld.de>" \
                  "\0" AMIGA_VERSION_STR;

unsigned char *outblock = NULL, *inblock = NULL;
unsigned short outblock_size = 0, inblock_size = 0;
unsigned short outblock_used = 0, inblock_actual_size = 0;
unsigned short outblock_base = 0, inblock_base = 0;
pc_file_t outfile;
pc_file_t infile;

char *basedir = NULL;
int verbose = 0;
int ignore_prot = 0;

char error_message[256];

enum { OVERWRITE_REQUEST, OVERWRITE_ALL, OVERWRITE_NONE } overwrite_mode = OVERWRITE_REQUEST;

int init_transferblocks( void )
{
  int i;

  outblock=(unsigned char*)malloc(TRANSFERBLOCKSIZE);
  inblock=(unsigned char*)malloc(TRANSFERBLOCKSIZE);
  if(outblock!=NULL)
    for(i=0;i<TRANSFERBLOCKSIZE;i++) outblock[i]=0;
  if(inblock!=NULL)
    for(i=0;i<TRANSFERBLOCKSIZE;i++) inblock[i]=0;
  if((inblock==NULL)||(outblock==NULL)) return 1;
  return 0;
}

int init_outgoing_file(const char*name)
{
  int fp;
  FILE *tfp;
  int size;
  
  outfile.size=0;
  outfile.checksum=0;
  outfile.fp=0;
  outfile.opened = 0;
  
#if defined AMIGA
  adf_mode = 0;

  if ( direct_disk_access &&
       strlen( name ) == 4 &&
       ( name[0] == 'd' || name[0] == 'D' ) &&
       ( name[1] == 'f' || name[1] == 'F' ) &&
       ( name[2] >= '0' || name[2] <= '3' ) &&
       name[3] == ':' ) {
      char *newname;
      char device_number = 0;

      newname = (char*)malloc( 256 );
      if ( newname == NULL ) {
#if defined GUI
          rtEZRequest( "No mem!", "Okay", NULL, (struct TagItem *)rttags );
#else
          printf( "No mem!\n" );
#endif
          return 1;
      }

      strcpy( newname, "dump.adf" );

      device_number = name[2] - '0';

#if defined GUI
      if ( ! rtGetStringA( newname, 255, "Enter name of adf file:", NULL, (struct TagItem *)rttags ) ) {
          free( newname );
          return 1;
      }
#else
      printf( "Enter name of adf file:" );
      fflush( stdout );
      scanf( "%255s", newname );
#endif
      
      if ( openDiskDevice( "trackdisk.device", device_number ) != 0 ) {
          long d = device_number;

#if defined GUI
          rtEZRequest( "Could not open trackdisk.device, unit %ld, skipping...", "Ok", NULL, (struct TagItem *)rttags, d );
#else
          printf( "Could not open trackdisk.device, unit %d, skipping...\n", device_number );
#endif

          free( newname );
          return 1;
      } else {
          outfile.size = 512 * 11 * 2;

          outfile.size *= ( max_track - min_track + 1 );

          outfile.name = newname;
          adf_mode = 1;
      }

      return 0;
  }
#endif

  tfp=fopen(name,"r");
  if(tfp!=NULL) {
    fseek(tfp,0,SEEK_END);
    size=ftell(tfp);
    fseek(tfp,0,SEEK_SET);
    outfile.size=size;
    fp=open(name,O_RDONLY);
    fclose(tfp);
    outfile.fp=fp;
    outfile.opened = 1;
  } else return 1;
  outfile.name=strdup(name);
  return 0;
}

int init_incoming_file( const char*name, int size )
{
  FILE *tfp;
  int fp;
#ifndef GUI
  char choose[8];
#endif
  int fileok;
#if defined AMIGA && defined GUI
  int res;
#endif
  short is_adf = 0;
  
  infile.size=0;
  infile.checksum=0;
  infile.fp=0;
  infile.name = NULL;
  infile.opened = 0;

#if defined AMIGA
  adf_mode = 0;

  /* accept file sizes for adf which are multiple of single-sided track */
  if ( ( size % ( 512 * 11 ) ) == 0 &&
       strlen( name ) > 4 &&
       memcmp( name + strlen( name ) - 4,
               ".adf", 4 ) == 0 ) {
      is_adf = 1;
  }
#endif

  if ( is_adf &&
#if defined AMIGA
       direct_disk_access
#else
       0
#endif
       ) {
#if defined AMIGA
      char device_number = 0;
#if defined GUI
      ULONG my_rttags[] = { RT_PubScrName, (ULONG)0,
                            RTEZ_DefaultResponse, 5,
                            RT_Underscore, '_',
                            TAG_END };
#endif

      /* ask for device */
      fileok = -1;

#if defined GUI
      // reapply pubscreen
      my_rttags[1] = rttags[1];

      res = rtEZRequest( "Receiving ADF disk image %s.\nSelect target device for writing or skip.", "DF_0|DF_1|DF_2|DF_3|Skip|Abort", NULL, (struct TagItem *)my_rttags, name );
      if ( res == 1 ) {
          device_number = 0;
          fileok = 0;
      } else if ( res == 2 ) {
          device_number = 1;
          fileok = 0;
      } else if ( res == 3 ) {
          device_number = 2;
          fileok = 0;
      } else if ( res == 4 ) {
          device_number = 3;
          fileok = 0;
      } else if ( res == 5 ) {
          fileok = -1;
      } else {
          fileok = -2;
      }
#else
      printf( "Receiving ADF disk image %s.\nSelect target device for writing or skip.\n\tdf0: (0), df1: (1), df2: (2), df3: (3)\n\tSkip (s) or abort (a):", name );
      fflush( stdout );
      scanf( "%4s", choose );
      if ( choose[0] == '0' ) {
          device_number = 0;
          fileok = 0;
      } else if ( choose[0] == '1' ) {
          device_number = 1;
          fileok = 0;
      } else if ( choose[0] == '2' ) {
          device_number = 2;
          fileok = 0;
      } else if ( choose[0] == '3' ) {
          device_number = 3;
          fileok = 0;
      } else if ( choose[0] == 's' ) {
          fileok = -1;
      } else {
          fileok = -2;
      }
#endif

      if ( fileok == 0 ) {
          if ( openDiskDevice( "trackdisk.device", device_number ) != 0 ) {
              long d = device_number;

#if defined GUI
              rtEZRequest( "Could not open trackdisk.device, unit %ld, skipping...", "Ok", NULL, (struct TagItem *)rttags, d );
#else
              printf( "Could not open trackdisk.device, unit %d, skipping...\n", device_number );
#endif

              fileok = -1;
          } else {
              adf_mode = 1;
          }
      }

      if ( fileok < 0 ) {
          return fileok;
      }
#else
      return -1;
#endif /* AMIGA */
  } else {
      char *newname, *basename, *tstr=NULL;

      newname = (char*)malloc( 256 );
      if ( newname == NULL ) {
#if defined AMIGA && defined GUI
          rtEZRequest( "No mem!", "Okay", NULL, (struct TagItem *)rttags );
#else
          printf( "No mem!\n" );
#endif
          return -2;
      }

      strcpy(newname,name);

      if(basedir!=NULL) {
          basename = strdup( basedir );
      } else {
          basename=strdup("");
      }

      fileok=0;
      do {
          if(tstr!=NULL) free(tstr);
          tstr=(char*)malloc(strlen(basename)+strlen(newname)+1);
          sprintf(tstr,"%s%s",basename,newname);
          tfp=fopen(tstr,"r");
          if(tfp!=NULL) {
              fclose(tfp);
              if ( overwrite_mode == OVERWRITE_ALL ) {
                  fileok = 1;
              } else if ( overwrite_mode == OVERWRITE_NONE ) {
                  fileok = -1;
              } else {
#if defined AMIGA && defined GUI
                  res = rtEZRequest( "File %s already exists!", "_Change Name|_Overwrite|Overwrite _all|_Skip|Skip all|Abort", NULL, (struct TagItem *)rttags, tstr );
                  if ( res == 2 ) {
                      fileok = 1;
                  } else if ( res == 3 ) {
                      fileok = 1;
                      overwrite_mode = OVERWRITE_ALL;
                  } else if ( res == 4 ) {
                      fileok = -1;
                  } else if ( res == 5 ) {
                      fileok = -1;
                      overwrite_mode = OVERWRITE_NONE;
                  } else if ( res == 0 ) {
                      fileok = -2;
                  } else {
                      if ( ! rtGetStringA( newname, 255, "Enter new name:", NULL, (struct TagItem *)rttags ) ) {
                          fileok = -2;
                      }
                  }
#else
                  printf("File %s already exists!\n\tOverwrite (o), Overwrite all (o!),\n\tSkip (s), Skip all(s!),\n\tchange (c) name or abort (a):",tstr);
                  fflush(stdout);
                  scanf( "%4s", choose );
                  if(choose[0]=='o') {
                      fileok=1;
                      if ( choose[1] == '!' ) overwrite_mode = OVERWRITE_ALL;
                  } else if ( choose[0] == 's' ) {
                      fileok = -1;
                      if ( choose[1] == '!' ) overwrite_mode = OVERWRITE_NONE;
                  } else if ( choose[0] == 'a' ) {
                      fileok = -2;
                  } else {
                      printf("Enter new name:");
                      fflush(stdout);
                      scanf( "%255s", newname );
                  }
#endif
              }
          } else fileok=1;
      } while(fileok==0);
      free( basename );
      if ( fileok < 0 ) {
          free( tstr );
          free( newname );
          return fileok;
      }
      fp=open(tstr,O_WRONLY|O_CREAT|O_TRUNC,0644);
      infile.fp=fp;
      if(fp<0) {
          free(tstr);
          free( newname );
          return 1;
      }
      free(tstr);

      infile.name = newname;
      infile.opened = 1;
  }

  return 0;
}

void close_outgoing_file( void )
{
    if ( outfile.opened != 0 ) {
        close(outfile.fp);
    } else {
#if defined AMIGA
        closeDiskDevice();
#endif
    }
}

void close_incoming_file( void )
{
    if ( infile.opened != 0 ) {
        close( infile.fp );
    } else {
#if defined AMIGA
        closeDiskDevice();
#endif
    }
}

void putint(unsigned char *buffer,int value)
{
  int b3,b2,b1,b0;
  b0=value&0xff;
  value>>=8;
  b1=value&0xff;
  value>>=8;
  b2=value&0xff;
  value>>=8;
  b3=value&0xff;
  *buffer++=(unsigned char)b3;
  *buffer++=(unsigned char)b2;
  *buffer++=(unsigned char)b1;
  *buffer++=(unsigned char)b0;
}

int getint(const unsigned char *buffer)
{
  int value;
  value=(int)*buffer++;
  value<<=8;
  value|=(int)*buffer++;
  value<<=8;
  value|=(int)*buffer++;
  value<<=8;
  value|=(int)*buffer++;
  return value;
}

int readtobuffer(unsigned char* buffer,int maxsize,int *return_checksum)
{
  int read_bytes,checksum,i;

  if ( outfile.opened ) {
      read_bytes =read(outfile.fp,buffer,maxsize);
  } else {
      read_bytes = maxsize;
  }

  checksum=0;
  for(i=0;i<read_bytes;i++) checksum+=(int)buffer[i];
  outfile.checksum+=checksum;
  if(return_checksum!=NULL) *return_checksum=checksum;
  return read_bytes;
}

int writefrombuffer(const unsigned char *buffer,int size,int *return_checksum)
{
  int written,checksum,i;

  if ( infile.opened ) {
      written = write(infile.fp,buffer,size);
  } else {
      written = size;
  }
  checksum=0;
  for(i=0;i<size;i++) checksum+=buffer[i];
  infile.checksum+=checksum;
  if(return_checksum!=NULL) *return_checksum=checksum;
  return written;
}

int min(int a, int b)
{
  if ( a < b ) return a;
  return b;
}

void *allocsafe( size_t size )
{
  void *ptr;
  
  ptr = malloc( size );
  if ( ptr == NULL ) {
    printf( "No Memory!\n" );
    exit( 5 );
  }
  return ptr;
}

void putuint16( unsigned char *buffer, unsigned short value )
{
  int b1,b0;
  b0=value&0xff;
  value>>=8;
  b1=value&0xff;
  *buffer++=(unsigned char)b1;
  *buffer++=(unsigned char)b0;
}

unsigned short getuint16( const unsigned char *buffer )
{
  unsigned short value;
  value=(int)*buffer++;
  value<<=8;
  value|=(int)*buffer++;
  return value;
}

void outputInt( unsigned short offset, int value )
{
    putint( outblock + outblock_base + offset, value );
}

int inputInt( unsigned short offset )
{
    return getint( inblock + inblock_base + offset );
}

unsigned char *outputData( unsigned short offset )
{
    return outblock + outblock_base + offset;
}

const unsigned char *inputData( unsigned short offset )
{
    return inblock + inblock_base + offset;
}

void setOutputSize( unsigned short size )
{
    outblock_used = size;
}

unsigned short inputSize( void )
{
    if ( inblock_base == 0 ) {
        return inblock_size;
    } else {
        return inblock_actual_size - inblock_base;
    }
}

void outputTrailingInt( int value )
{
    if ( outblock_base == 0 ) {
        putint( outblock + outblock_size - 4, value );
    } else {
        // used is without base
        putint( outblock + outblock_base + outblock_used - 4, value );
    }
}

int inputTrailingInt( void )
{
    if ( inblock_base == 0 ) {
        return getint( inblock + inblock_size - 4 );
    } else {
        // actual size includes the base
        return getint( inblock + inblock_actual_size - 4 );
    }
}

unsigned short maxOutputSize( void )
{
    return outblock_size - outblock_base;
}

unsigned short maxInputSize( void )
{
    return inblock_size - inblock_base;
}
