/* apccomm_am_main.c
   Part of "APCComm" 
   Copyright (C) 2000-2025 Ralf Hoffmann
   Contact: ralf@boomerangsworld.de

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
  
*/

#include "apccomm_am.h"
#include "apccomm_am_tr.h"
#include "apccomm_all.h"
#include "apccomm_am_main.h"

#include <ctype.h>
#include <signal.h>

#include <workbench/workbench.h>
#include <workbench/startup.h>
#include <dos/dos.h>
#include <dos/var.h>
#include <proto/dos.h>
#include <proto/icon.h>
#include <proto/wb.h>
#include <exec/types.h>
#include <exec/libraries.h>
#include <proto/intuition.h>
#include <proto/exec.h>
#include <proto/misc.h>
#include <proto/reqtools.h>
#include <exec/exec.h>
#include <resources/misc.h>

struct ReqToolsBase *ReqToolsBase = NULL;
struct apccomm_transfer_config config;

#ifdef GUI

#include "apccomm_gui.h"
#include "lvhandling.h"

#ifdef _DCC
/* dcc has strcasecmp but no prototype, but stricmp does the same
   and is decleared even the docs so I will use it for dcc
*/
#define strcasecmp stricmp
#endif

int selected_lvc = -1;
#endif

enum guimodes mode = GUIMODE_RUN;
short StartFromWB = 0;
BOOL freePubScreenName = FALSE;


int direct_disk_access = 0;
int adf_write_verify = 0;
int min_track = 0;
int max_track = 79;
short adf_mode = 0;  /* 1 if we are currently reading/writing an adf file */

#ifdef GUI

char *my_getcwd( void )
{
  int cwdsize;
  char *cwd;

  cwdsize = 1024;
  cwd = (char*) malloc( cwdsize );
  if ( cwd != NULL ) {
    while ( getcwd( cwd, cwdsize ) == NULL ) {
      if ( errno == ERANGE ) {
        free( cwd );
        cwdsize *= 2;
        cwd = (char*)malloc( cwdsize );
        if ( cwd == NULL ) break;
      } else {
        free( cwd );
        cwd = NULL;
        break;
      }
    }
  }
  return cwd;
}

void setRSG( const char *text )
{
  struct Gadget *sg;

  if ( text == NULL ) return;
  sg = MainWindowGadgets[ GD_rdsg ];

  GT_SetGadgetAttrs( sg, MainWindowWnd, NULL, GTST_String, text, TAG_END, 0 );
}

void requestReceiveDir( void )
{
  struct rtFileRequester *filereq;
  char filename[109];
  struct Gadget *sg;

  sg = MainWindowGadgets[ GD_rdsg ];

  strcpy( filename, "" );
  if ( filereq = rtAllocRequestA( RT_FILEREQ, NULL ) ) {
    rtChangeReqAttr( filereq, RTFI_Dir, GetString( sg ), TAG_END );
    if ( rtFileRequest( filereq, filename, "Pick a directory", RTFI_Flags, FREQF_NOFILES, RT_PubScrName, (ULONG)PubScreenName, TAG_END ) ) {
      setRSG( filereq->Dir );
    }
    rtFreeRequest( filereq );
  } else {
    rtEZRequest( "Out of memory!", "Oh boy!", NULL, (struct TagItem *)rttags );
  }
}

void initRSG( void )
{
  char *cwd;
  
  cwd = my_getcwd();
  setRSG( cwd );
  if ( cwd != NULL ) free( cwd );
}

void requestAndAddFiles( void )
{
  struct rtFileRequester *filereq;
  struct rtFileList *flist, *tempflist;
  char filename[109];
  char *tstr, *dir;
  struct Gadget *lv;

  lv = MainWindowGadgets[ GD_sflv ];
  if ( filereq = rtAllocRequestA( RT_FILEREQ, NULL ) ) {
    filename[0] = 0;

    flist = rtFileRequest( filereq, filename, "Pick some files", RTFI_Flags, FREQF_MULTISELECT | FREQF_SELECTDIRS, RT_PubScrName, (ULONG)PubScreenName, TAG_END );
    if ( flist ) {
      if ( strlen( filereq->Dir ) < 1 ) {
        dir = my_getcwd();
      } else {
        dir = strdup( filereq->Dir );
      }
      
      tempflist = flist;
      while ( tempflist ) {
        tstr = (char*)malloc( strlen( dir ) + 1 + strlen( tempflist->Name ) + 1 );
        if ( dir[ strlen( dir ) - 1 ] == ':' ) {
          sprintf( tstr, "%s%s", dir, tempflist->Name );
        } else {
          sprintf( tstr, "%s/%s", dir, tempflist->Name );
        }
        addLVC( tstr );
        updateLV( lv, MainWindowWnd );
        free( tstr );
        tempflist = tempflist->Next;
      }
      rtFreeFileList( flist );
    }

    rtFreeRequest( filereq );
  } else {
    rtEZRequest( "Out of memory!", "Oh boy!", NULL, (struct TagItem *)rttags );
  }
}

void removeSelectedEntry( void )
{
  struct Gadget *lv;
  int s;

  if ( selected_lvc < 0 ) return;
  if ( lvcs == NULL ) return;

  lv = MainWindowGadgets[ GD_sflv ];

  removeLVCAtPos( selected_lvc );
  updateLV( lv, MainWindowWnd );
  s = getLVCSize();
  if ( selected_lvc >= s ) {
    selected_lvc = s - 1;
  }
  GT_SetGadgetAttrs( lv, MainWindowWnd, NULL, GTLV_Selected, selected_lvc, TAG_END );
}

void applySLVSG( void )
{
  struct Gadget *sg;
  struct Gadget *lv;
  char *tstr;

  sg = MainWindowGadgets[ GD_slvsg ];
  lv = MainWindowGadgets[ GD_sflv ];

  if ( selected_lvc < 0 ) {
    /* nothing selected so add SG content */
    tstr = GetString( sg );
    if ( strlen( tstr ) > 0 ) {
      addLVC( tstr );
      updateLV( lv, MainWindowWnd );
    }
  } else {
    /* change value at selected position */
    tstr = GetString( sg );
    if ( strlen( tstr ) > 0 ) {
      changeLVC( selected_lvc, tstr );
      updateLV( lv, MainWindowWnd );
    }
  }
}

void showAbout( void )
{
  if ( rtEZRequest( (char*)aboutStr, "Okay|More", NULL, (struct TagItem *)rttags, (long)MAJOR, (long)MINOR, (long)PATCH ) == 0 ) {
    rtEZRequest( (char*)aboutStr2, "Okay", NULL, (struct TagItem *)rttags );
  }
}

static void updateSerialSpeedGadget( void )
{
    switch ( config.serial_speed ) {
        case 9600:
            GT_SetGadgetAttrs( MainWindowGadgets[ GD_serialspeedcyb ], MainWindowWnd, NULL, GTCY_Active, 0, TAG_END, 0 );
            break;
        case 19200:
            GT_SetGadgetAttrs( MainWindowGadgets[ GD_serialspeedcyb ], MainWindowWnd, NULL, GTCY_Active, 1, TAG_END, 0 );
            break;
        case 38400:
            GT_SetGadgetAttrs( MainWindowGadgets[ GD_serialspeedcyb ], MainWindowWnd, NULL, GTCY_Active, 2, TAG_END, 0 );
            break;
        case 115200:
            GT_SetGadgetAttrs( MainWindowGadgets[ GD_serialspeedcyb ], MainWindowWnd, NULL, GTCY_Active, 4, TAG_END, 0 );
            break;
        case 57600:
        default:
            GT_SetGadgetAttrs( MainWindowGadgets[ GD_serialspeedcyb ], MainWindowWnd, NULL, GTCY_Active, 3, TAG_END, 0 );
            config.serial_speed = 57600;
            break;
    }
}

void portSelected( UWORD code )
{
    switch ( code ) {
        case 0: // parallel mode
            GT_SetGadgetAttrs( MainWindowGadgets[ GD_serialspeedcyb ], MainWindowWnd, NULL, GA_Disabled, TRUE, TAG_END, 0 );
            config.mode = APCCOMM_PARALLEL;
            break;
        case 1: // serial mode
            GT_SetGadgetAttrs( MainWindowGadgets[ GD_serialspeedcyb ], MainWindowWnd, NULL, GA_Disabled, FALSE, TAG_END, 0 );
            config.mode = APCCOMM_SERIAL;

            updateSerialSpeedGadget();
            break;
        default:
            break;
    }
}

void speedSelected( UWORD code )
{
    switch ( code ) {
        case 0:
            config.serial_speed = 9600;
            break;
        case 1:
            config.serial_speed = 19200;
            break;
        case 2:
            config.serial_speed = 38400;
            break;
        case 4:
            config.serial_speed = 115200;
            break;
        case 3:
        default:
            config.serial_speed = 57600;
            break;
    }
}

#endif

void CleanUpHandler( void )
{
#ifdef GUI
  if ( ( freePubScreenName == TRUE ) && ( PubScreenName != NULL ) ) {
    free( PubScreenName );
  }

  if ( ReqToolsBase ) {
      CloseLibrary( (struct Library *)ReqToolsBase );
      ReqToolsBase = NULL;
  }
#endif

  if ( am_cleanup != NULL ) {
      am_cleanup();
  }
}

#define os20str_alert "\0\330\17This program needs OS 2.0!\0\1\0\350\37Press any mousebutton\0\0"

static void sighandler( int sig )
{
    connect_cancel = 1;
    receive_cancel = 1;
    send_cancel = 1;
}

static int save_settings( short permanent,
                          const struct apccomm_transfer_config *config,
                          int ignore_prot,
                          int verbose )
{
    int retval = 0;

    if ( osversion < 36 ) {
        sprintf( error_message, "saving settings is not support is OS<2.0" );
        return -EINVAL;
    }

    char *buf = malloc( 1024 );
    if ( ! buf ) {
        sprintf( error_message, "no memory for config buffer" );
        return -ENOMEM;
    }

    if ( config->mode == APCCOMM_PARALLEL ) {
        sprintf( buf, "PORT=PARALLEL" );
    } else {
        sprintf( buf, "PORT=SERIAL SPEED=%d", config->serial_speed );
    }

    int res = SetVar( "apccomm.prefs", buf, strlen( buf ) + 1, GVF_GLOBAL_ONLY );
    if ( res == 0 ) {
        free( buf );
        return -ENOMEM;
    }

    if ( permanent ) {
        FILE *i_fh = fopen( "ENV:apccomm.prefs", "r" );
        FILE *o_fh = fopen( "ENVARC:apccomm.prefs", "w" );

        if ( i_fh ) {
            if ( o_fh ) {
                char buf[16];
                while ( fread( buf, 1, 1, i_fh ) == 1 ) {
                    if ( fwrite( buf, 1, 1, o_fh ) != 1 ) {
                        break;
                    }
                }

                if ( feof( i_fh ) && ! ferror( o_fh ) ) {
                } else {
                    sprintf( error_message, "Failed to write permanent settings to ENVARC:apccomm.prefs" );
                    unlink( "ENVARC:apccomm.prefs" );
                }

                fclose( o_fh );
            } else {
                sprintf( error_message, "Failed to open ENVARC variable ENVARC:apccomm.prefs" );
                retval = -ENOENT;
            }
            fclose( i_fh );
        } else {
            sprintf( error_message, "Failed to open ENV variable ENV:apccomm.prefs" );
        }
    }

    free( buf );

    return retval;
}

static int load_settings( struct apccomm_transfer_config *config,
                          int *ignore_prot,
                          int *verbose )
{
    if ( osversion < 36 ) {
        return -EINVAL;
    }

    char *buf = malloc( 1024 );
    if ( ! buf ) {
        sprintf( error_message, "no memory for config buffer" );
        return -ENOMEM;
    }

    int l = GetVar( "apccomm.prefs", buf, 1024, GVF_GLOBAL_ONLY );
    if ( l < 0 ) {
        free( buf );
        return -ENOENT;
    }
    if ( l + 1 >= 1024 ) {
        free( buf );
        return -ENOMEM;
    }

    buf[l] = '\0';

    char *token;
    for ( token = strtok( buf, " " );
          token != NULL;
          token = strtok( NULL, " " ) ) {
        if ( strlen( token ) < 2 ) continue;

        if ( strncmp( token, "PORT=", 5 ) == 0 ) {
            if ( strcmp( token + 5, "PARALLEL" ) == 0 ) {
                config->mode = APCCOMM_PARALLEL;
            } else if ( strcmp( token + 5, "SERIAL" ) == 0 ) {
                config->mode = APCCOMM_SERIAL;
            }
        } else if ( strncmp( token, "SPEED=", 6 ) == 0 ) {
            int s = atoi( token + 6 );
            switch ( s ) {
                case 9600:
                case 19200:
                case 38400:
                case 57600:
                case 115200:
                    config->serial_speed = s;
                    break;
                default:
                    break;
            }
        }
    }

    free( buf );

    return 0;
}

void menuSettings( short permanent )
{
    if ( save_settings( permanent, &config, ignore_prot, verbose ) != 0 ) {
#ifdef GUI
        rtEZRequest( "Failed to store settings:\n%s\n", "Okay", NULL, (struct TagItem *)rttags, error_message );
#endif
    }
}

int main(int argc,char **argv)
{
  int s1;
  int o1,o2,o3, res;
  short *realargs = NULL;
#ifdef GUI
  struct Gadget *lv;
  struct Node *node;
#endif
  int store_settings = 0;

  osversion = SysBase->LibNode.lib_Version;
  atexit( CleanUpHandler );
#ifdef GUI
  if ( osversion < 36 ) {
    DisplayAlert( RECOVERY_ALERT, os20str_alert, 45 );
    exit( 1 );
  }

  ReqToolsBase = (struct ReqToolsBase *)OpenLibrary( "reqtools.library", 0 );
  if ( ! ReqToolsBase ) {
      struct EasyStruct reqstr = { sizeof( struct EasyStruct ),
                                   0,
                                   "Missing library",
                                   "Library reqtools.library missing",
                                   "Ok" };
      EasyRequestArgs( NULL, &reqstr, NULL, NULL );
      exit( 1 );
  }
#endif

  signal( SIGINT, sighandler );

  realargs=(short*)malloc(sizeof(short)*argc);
  for(s1=0;s1<argc;s1++) {
    realargs[s1]=1;
  }

  if ( ! StartFromWB ) {
      memset( &config, 0, sizeof( config ) );
      config.mode = APCCOMM_PARALLEL;

      load_settings( &config,
                     &ignore_prot,
                     &verbose );
  }

  if(argc>1) {
    for(s1=1;s1<argc;s1++) {
      if ( ( strncmp( argv[s1], "-h", 2 ) == 0 ) ||
	   ( strncmp( argv[s1], "--help", 6 ) == 0 ) ||
	   ( strcmp( argv[s1], "?" ) == 0 ) ) {
#ifdef GUI
        printf("\nAPCComm GUI Version by Ralf Hoffmann\n");
#else
        printf("\nAPCComm by Ralf Hoffmann\n");
#endif
        printf("  Usage: %s [Option]... [<filename>]...\n  Transfer given files to the PC\n\n",argv[0]);
        printf("   -h, --help\t\t\tShow this help\n");
        printf("   -v, --version\t\tShow program version\n");
	printf("   --license\t\t\tShow program license\n" );
        printf("   -V, --verbose\t\tverbose output\n");
        printf("   -q, --quiet\t\t\tno output (except errors)\n");
        printf("   -i\t\t\t\tignore file protection for receiving\n");
        printf("   -d, --disk\t\t\tread or write adf files directly from/to\n\t\t\t\tdisk drive\n\t\t\t\t(asks before actually accessing drive)\n");
        printf("   -t, --track min,max\t\tread only from min to max track\n");
        printf("   --verify\t\t\tverify written adf image\n");
        printf("   --parallel\t\t\tUse the parallel port for transmission\n");
        printf("   --serial\t\t\tUse the serial port for transmission\n");
        printf("   --speed <mode>\t\tselect serial speed (slow,normal,fast or\n\t\t\t\ta number)\n");
        printf("   --keep-settings\t\tstore some basic settings in ENV variable\n");
        printf("   --save-settings\t\tsave some basic settings for permanent use\n");
#ifdef GUI
        printf("   -s, --send\t\t\timmediately start sending\n");
        printf("   -r, --receive\t\timmediately start receiving\n");
        printf("   --pubscreen PUBSCREENNAME\tUse given pubscreen for gui\n\n");
#endif
#ifdef GUI
        printf("  When starting without args, APCComm will open a window which will let\n  you choose the files to send or where to store received files.\n");
#else
        printf("  When starting without args, APCComm will receive files from the PC\n");
#endif
        exit(0);
      } else if((strncmp(argv[s1],"-v",2)==0)||(strncmp(argv[s1],"--version",6)==0)) {
        printf("\nAPCComm by Ralf Hoffmann\n  Version %d.%d.%d\n",MAJOR,MINOR,PATCH);
        printf("\n  Program for transfering files between Amiga and Linux\n");
#ifdef GUI
        printf("  GUI Version\n");
#endif
	printf( "\n  This is free software with ABSOLUTELY NO WARRANTY\n" );
	printf( "  For details use --license\n\n" );
        printf( "%s", aboutStr2 );
        printf( "\n" );
        exit(0);
      } else if ( strcmp( argv[s1], "--license" ) == 0 ) {
	printf( aboutStr, MAJOR, MINOR, PATCH );
	printf( "\n" );
	exit( 0 );
      } else if( ( strncmp(argv[s1],"--verbose",strlen("--verbose"))==0 ) ||
		 ( strncmp(argv[s1],"-V",strlen("-V"))==0 ) ) {
	verbose = 1;
	realargs[s1]=0;
      } else if( ( strncmp(argv[s1],"--quiet",strlen("--quiet"))==0 ) ||
		 ( strncmp(argv[s1],"-q",strlen("-q"))==0 ) ) {
	verbose = -1;
	realargs[s1]=0;
      } else if( strncmp(argv[s1],"-i",strlen("-i"))==0 ) {
	ignore_prot = 1;
	realargs[s1]=0;
      } else if ( strcmp( argv[s1], "--keep-settings" ) == 0 ) {
          store_settings = 1;
          realargs[s1]=0;
      } else if ( strcmp( argv[s1], "--save-settings" ) == 0 ) {
          store_settings = 2;
          realargs[s1]=0;
      } else if ( ( strncmp( argv[s1], "--disk", strlen( "--disk" ) ) == 0 ) ||
                  ( strncmp( argv[s1], "-d", strlen( "-d" ) ) == 0 ) ) {
          direct_disk_access = 1;
          realargs[s1]=0;
      } else if ( ( strncmp( argv[s1], "--verify", strlen( "--verify" ) ) == 0 ) ) {
          adf_write_verify = 1;
          realargs[s1]=0;
      } else if ( ( strncmp( argv[s1], "--serial", strlen( "--serial" ) ) == 0 ) ) {
          config.mode = APCCOMM_SERIAL;
          realargs[s1]=0;
      } else if ( ( strncmp( argv[s1], "--parallel", strlen( "--parallel" ) ) == 0 ) ) {
          config.mode = APCCOMM_PARALLEL;
          realargs[s1]=0;
      } else if ( ( strncmp( argv[s1], "--track", strlen( "--track" ) ) == 0 ) ||
                  ( strncmp( argv[s1], "-t", strlen( "-t" ) ) == 0 ) ) {
          realargs[s1]=0;

          if ( ( s1 + 1 ) < argc ) {
            s1++;
            if ( sscanf( argv[s1], "%d,%d", &min_track, &max_track ) == 2 ) {
                realargs[s1]=0;
            } else {
                fprintf( stderr, "track argument missing!\n" );
                exit( 1 );
            }
          } else {
            fprintf( stderr, "track argument missing!\n" );
            exit( 1 );
          }
      } else if ( strncmp( argv[s1], "--speed", strlen( "--speed" ) ) == 0 ) {
          realargs[s1]=0;
          const char *speed_name = NULL;
          if ( strncmp( argv[s1], "--speed=", strlen( "--speed=" ) ) == 0) {
              speed_name = argv[s1] + strlen( "--speed=" );
          } else {
              if ( ( s1 + 1 ) < argc ) {
                  s1++;
                  speed_name = argv[s1];
                  realargs[s1]=0;
              }
          }

          if ( speed_name != NULL ) {
              config.mode = APCCOMM_SERIAL;
              if ( strcmp( speed_name, "slow" ) == 0 ) {
                  config.serial_speed = 9600;
              } else if ( strcmp( speed_name, "fast" ) == 0 ) {
                  config.serial_speed = 115200;
              } else {
                  config.serial_speed = 57600;

                  if ( strlen( speed_name ) > 0 &&
                       isdigit( speed_name[0] ) ) {
                      int v = atoi( speed_name );

                      switch ( v ) {
                          case 9600:
                          case 19200:
                          case 38400:
                          case 57600:
                          case 115200:
                              config.serial_speed = v;
                              break;
                          default:
                              fprintf( stderr, "Unsupported speed %s\n", speed_name );
                              exit( 1 );
                      }
                  }
              }
          } else {
              fprintf( stderr, "Speed argument missing!\n" );
              exit( 1 );
          }
#ifdef GUI
      } else if ( ( strncmp( argv[s1], "--send", strlen( "--send" ) ) == 0 ) ||
		  ( strncmp( argv[s1], "-s", strlen( "-s" ) ) == 0 ) ) {
        mode = GUIMODE_SEND;
	realargs[s1]=0;
      } else if ( ( strncmp( argv[s1], "--receive", strlen( "--receive" ) ) == 0 ) ||
		  ( strncmp( argv[s1], "-r", strlen( "-r" ) ) == 0 ) ) {
        mode = GUIMODE_RECEIVE;
	realargs[s1]=0;
      } else if ( strncmp( argv[s1], "--pubscreen", strlen( "--pubscreen" ) ) == 0 ) {
	realargs[s1]=0;
        if ( strncmp( argv[s1], "--pubscreen=", strlen( "--pubscreen=" ) ) == 0) {
          PubScreenName = argv[s1] + strlen( "--pubscreen=" );
        } else {
          if ( ( s1 + 1 ) < argc ) {
            s1++;
            PubScreenName = argv[s1];
            realargs[s1]=0;
          } else {
            fprintf( stderr, "Pubscreen argument missing!\n" );
            exit( 1 );
          }
        }
#endif
      }
    }
  }

  if ( store_settings > 0 ) {
      if ( save_settings( store_settings == 2, &config, ignore_prot, verbose ) != 0 ) {
#ifdef GUI
          rtEZRequest( "Failed to store settings:\n%s\n", "Okay", NULL, (struct TagItem *)rttags, error_message );
#else
          fprintf( stderr, "Failed to store settings:\n%s\n", error_message );
#endif
          exit( 1 );
      }
  }

  if(init_transferblocks()!=0) {
    fprintf(stderr,"Not enough mem in main()!\nExiting\n");
    exit(1);
  }

#ifdef GUI
  /* Try to setup screen, first the given pubscreen and if this fails, the
     default pubscreen (in most cases Workbench) */
  while ( SetupScreen() != 0 ) {
    if ( PubScreenName != NULL ) {
      rtEZRequest( "Couldn't use pubscreen %s!\nTrying default pubscreen..", "Okay", NULL, (struct TagItem *)rttags, PubScreenName );
      if ( freePubScreenName == TRUE ) free( PubScreenName );
      freePubScreenName = FALSE;
      PubScreenName = NULL;
    } else {
      rtEZRequest( "Couldn't use default pubscreen!", "Okay", NULL, (struct TagItem *)rttags );
      mode = GUIMODE_QUIT;
      break;
    }
  }
  
  rttags[1] = (ULONG)PubScreenName;

  initLVCList();
  for ( s1 = 1; s1 < argc; s1++ ) {
    if ( realargs[s1] == 1 ) {
      addLVC( argv[s1] );
    }
  }
  if ( mode == GUIMODE_RUN ) {
    OpenMainWindowWindow();
    lv = MainWindowGadgets[ GD_sflv ];
    
    updateLV( lv, MainWindowWnd );
    initRSG();
    GT_SetGadgetAttrs( MainWindowGadgets[ GD_ipcb ], MainWindowWnd, NULL, GTCB_Checked, ignore_prot, TAG_END, 0 );
    GT_SetGadgetAttrs( MainWindowGadgets[ GD_adfmodecb ], MainWindowWnd, NULL, GTCB_Checked, direct_disk_access, TAG_END, 0 );
    GT_SetGadgetAttrs( MainWindowGadgets[ GD_verifycb ], MainWindowWnd, NULL, GTCB_Checked, adf_write_verify, TAG_END, 0 );

    if ( config.mode == APCCOMM_SERIAL ) {
        GT_SetGadgetAttrs( MainWindowGadgets[ GD_portcyb ], MainWindowWnd, NULL, GTCY_Active, 1, TAG_END, 0 );
        GT_SetGadgetAttrs( MainWindowGadgets[ GD_serialspeedcyb ], MainWindowWnd, NULL, GA_Disabled, FALSE, TAG_END, 0 );
        updateSerialSpeedGadget();
    } else {
        GT_SetGadgetAttrs( MainWindowGadgets[ GD_portcyb ], MainWindowWnd, NULL, GTCY_Active, 0, TAG_END, 0 );
        GT_SetGadgetAttrs( MainWindowGadgets[ GD_serialspeedcyb ], MainWindowWnd, NULL, GA_Disabled, TRUE, TAG_END, 0 );
    }

    while ( mode == GUIMODE_RUN ) {
      WaitPort( MainWindowWnd->UserPort );
      HandleMainWindowIDCMP();
    }
    basedir = (char*)malloc( strlen( GetString( MainWindowGadgets[ GD_rdsg ] ) ) + 1 + 1 );
    strcpy( basedir, GetString( MainWindowGadgets[ GD_rdsg ] ) );
    s1 = strlen( basedir );
    if ( s1 > 0 ) {
      if ( ( basedir[ s1 - 1 ] != ':' ) &&
           ( basedir[ s1 - 1 ] != '/' ) ) {
        basedir[ s1 ] = '/';
        basedir[ s1 + 1 ] = '\0';
      }
    }
    CloseMainWindowWindow();
  }
#else
  mode = GUIMODE_RECEIVE;
  for ( s1 = 1; s1 < argc; s1++ ) {
      if ( realargs[ s1 ] == 1 ) {
          addSendListElem( argv[s1] );
          mode = GUIMODE_SEND;
      }
  }
#endif

  if ( mode == GUIMODE_QUIT ) {

#ifdef GUI
    deleteLVCList();
#endif

  } else {

#ifdef GUI
    node = GetHead( lvcs );
    while ( node != NULL ) {
      addSendListElem( node->ln_Name );
      node = GetSucc( node );
    }
    deleteLVCList();
#endif
    
    if ( ( mode == GUIMODE_SEND ) && ( SendListIsEmpty() == 1 ) ) {

#ifdef GUI
      rtEZRequest( "Nothing to send!", "Okay", NULL, (struct TagItem *)rttags );
#else
      printf( "Nothing to send!\n" );
#endif

    } else {

      if ( am_init_port( &config ) != 0 ) {
#ifdef GUI
	rtEZRequest( error_message, "Okay", NULL, (struct TagItem*)rttags );
#else
	fprintf( stderr, "%s\n", error_message );
#endif
      } else {

#ifdef GUI
	OpenConnectWindowWindow();
#else
	if ( ! BE_QUIET ) printf("Waiting for PC:");
	fflush(stdout);
#endif
	
	if ( am_sync() != 0 ) {
#ifdef GUI
	    rtEZRequest( "Syncing failed:\n%s\nAborting!", "Okay", NULL, (struct TagItem *)rttags, error_message );
#else
            fprintf( stderr, "Syncing failed:\n%s\nAborting...\n", error_message );
#endif
            connect_cancel = 1;
        }
	
#ifdef GUI
	CloseConnectWindowWindow();
#else
	if ( ! connect_cancel && ! BE_QUIET ) printf("\nConnection established\n");
#endif
	if ( connect_cancel == 0 ) {
	  
	  /* Prepare the outblock */
      outputInt( 0, PROTOCOL_MAJOR );
      outputInt( 4, PROTOCOL_MINOR );
      outputInt( 8, PROTOCOL_PATCH );
	  if ( mode == GUIMODE_SEND ) outputInt( 12, 1 );
	  else outputInt( 12, 0 );
      setOutputSize( 16 );
	  
	  /* Inform the other side */
	  res = am_transferblock();
	  
	  /* Now check the other version */
      o1 = inputInt( 0 );
      o2 = inputInt( 4 );
      o3 = inputInt( 8 );
	  if( res != 0 || (o1!=PROTOCOL_MAJOR) || (o2!=PROTOCOL_MINOR) ) {
#ifdef GUI
          if ( res != 0 ) {
              rtEZRequest( "Error during initial data transfer:\n%s", "Okay", NULL, (struct TagItem *)rttags, error_message );
          } else {
              rtEZRequest( "Wrong version at the other side!\n"
                           "  Expected %ld.%ld.x, got %ld.%ld.%ld", "Okay", NULL, (struct TagItem *)rttags, PROTOCOL_MAJOR, PROTOCOL_MINOR, o1, o2, o3 );
          }
#else
          if ( res != 0 ) {
              printf("Error during initial data transfer:\n%s\n", error_message );
          } else {
              printf("Wrong version at the other side!\n");
              printf("  Expected %d.%d.x ,  got %d.%d.%d\n",PROTOCOL_MAJOR,PROTOCOL_MINOR,o1,o2,o3);
          }
#endif
	  } else {
	    
	    /* Version ok, sending conflict ? */
        o1 = inputInt( 12 );
	    if( (o1==1) && ( mode == GUIMODE_SEND ) ) {
#ifdef GUI
	      rtEZRequest( "Both sides want to send!\n"
			   "  This is not supported!", "Okay", NULL, (struct TagItem *)rttags );
#else
	      printf("Both sides want to send!\n");
	      printf("  This is not supported!\n");
#endif
	    } else {
	      
	      if( (o1==0) && ( mode != GUIMODE_SEND ) ) {
#ifdef GUI
		rtEZRequest( "Nothing to do!", "Okay", NULL, (struct TagItem *)rttags );
#else
		printf("Nothing to do!\n");
#endif
	      } else {
		
		if ( mode == GUIMODE_SEND ) {
		  send();
		} else {
		  receive();
		}
	      } /* ! if( (o1==0) && (sendargs==0) ) */
	    } /* ! if( (o1==1) && (sendargs>0) ) */
	  } /* ! if( (o1!=MAJOR) || (o2!=MINOR) ) */
	} /* if ( connect_cancel == 0 ) */
      } /* if ( initParPort() != 0 ) */
    } /* ! if ( ( mode == GUIMODE_SEND ) && ( SendListIsEmpty() == 1 ) ) */
  } /* if ( mode == GUIMODE_QUIT ) */

  freeSendList();
#ifdef GUI
  CloseDownScreen();
#endif

  if ( basedir != NULL ) free( basedir );
  if ( inblock != NULL ) free( inblock );
  if ( outblock != NULL ) free( outblock );
  if ( realargs != NULL ) free( realargs );
  return 0;
}

#ifdef GUI
int wbmain( struct WBStartup *wbs )
{
  struct DiskObject *dob;
  BPTR oldlock;
  char *tstr;
  int i;
  
  osversion = SysBase->LibNode.lib_Version;
  if ( osversion < 36 ) {
    /* I think there is no easy way to show this message
       when started from WB */
    DisplayAlert( RECOVERY_ALERT, os20str_alert, 45 );
    exit( 1 );
  }

  memset( &config, 0, sizeof( config ) );
  config.mode = APCCOMM_PARALLEL;

  load_settings( &config,
                 &ignore_prot,
                 &verbose );

  if ( wbs->sm_NumArgs > 0 ) {
    oldlock = CurrentDir( (BPTR)wbs->sm_ArgList[0].wa_Lock );
    dob = GetDiskObject( wbs->sm_ArgList[0].wa_Name );
    if ( dob != NULL ) {
      if ( dob->do_ToolTypes != NULL ) {
        if ( ( tstr = FindToolType( dob->do_ToolTypes, "PUBSCREEN" ) ) != NULL ) {
          PubScreenName = strdup( tstr );
          freePubScreenName = TRUE;
        }
        if ( ( tstr = FindToolType( dob->do_ToolTypes, "VERBOSE_LEVEL" ) ) != NULL ) {
          i = atoi( tstr );
          if ( i == 0 ) verbose = -1;
          else if ( i == 2 ) verbose = 1;
          else verbose = 0;
        }
        if ( ( tstr = FindToolType( dob->do_ToolTypes, "IGNORE_PROT" ) ) != NULL ) {
          if ( ( strcasecmp( tstr, "true" ) == 0 ) ||
               ( strcasecmp( tstr, "yes" ) == 0 ) ||
               ( strcasecmp( tstr, "on" ) == 0 ) ||
               ( strcasecmp( tstr, "1" ) == 0 ) ) {
            ignore_prot = 1;
          } else ignore_prot = 0;
        }

        if ( ( tstr = FindToolType( dob->do_ToolTypes, "PORT" ) ) != NULL ) {
            if ( strcasecmp( tstr, "parallel" ) == 0 ) {
                config.mode = APCCOMM_PARALLEL;
            } else if ( strcasecmp( tstr, "serial" ) == 0 ) {
                config.mode = APCCOMM_SERIAL;
            }
        }

        if ( ( tstr = FindToolType( dob->do_ToolTypes, "SPEED" ) ) != NULL ) {
            if ( strcasecmp( tstr, "slow" ) == 0 ||
                 strcasecmp( tstr, "9600" ) == 0 ) {
                config.mode = APCCOMM_SERIAL;
                config.serial_speed = 9600;
            } else if ( strcasecmp( tstr, "19200" ) == 0 ) {
                config.mode = APCCOMM_SERIAL;
                config.serial_speed = 19200;
            } else if ( strcasecmp( tstr, "38400" ) == 0 ) {
                config.mode = APCCOMM_SERIAL;
                config.serial_speed = 38400;
            } else if ( strcasecmp( tstr, "normal" ) == 0 ||
                        strcasecmp( tstr, "57600" ) == 0 ) {
                config.mode = APCCOMM_SERIAL;
                config.serial_speed = 57600;
            } else if ( strcasecmp( tstr, "fast" ) == 0 ||
                        strcasecmp( tstr, "115200" ) == 0 ) {
                config.mode = APCCOMM_SERIAL;
                config.serial_speed = 115200;
            }
        }
      }
      FreeDiskObject( dob );
    }
    CurrentDir( oldlock );
  }
  StartFromWB = 1;
  return main( 1, NULL );  /* now call normal */
}
#endif
