/* apccomm_am_tr.c
   Part of "APCComm" 
   Copyright (C) 2000-2023 Ralf Hoffmann
   Contact: ralf@boomerangsworld.de

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
  
*/

#include "apccomm_am_tr.h"
#include "apccomm_am.h"
#include "apccomm_am_main.h"

#ifdef GUI
#include "apccomm_gui.h"
#endif

#include <exec/types.h>
#include <exec/memory.h>
#include <exec/devices.h>
#include <exec/io.h>
#include <devices/trackdisk.h>
#include <clib/alib_protos.h>
#include <clib/exec_protos.h>

/* must be multiple of 512
 * 512 * 11 is one single sided track
 */
#define DISK_IO_BUFFER_SIZE ( 512 * 11 )
static short io_pending = 0;

struct disk_buffer {
    UBYTE *chip_buffer[2];
    short chip_buffer_offset[2];
    short active_buffer;
    int current_io_offset;
    int max_io_offset;
    int total_size;
    int total_read_bytes;
};

static int finalizeDisk( void )
{
    dd_io_req->io_Command = CMD_UPDATE;
    dd_io_req->io_Length = 0;
    dd_io_req->io_Data = NULL;

    DoIO( (struct IORequest *) dd_io_req );

    dd_io_req->io_Command = TD_MOTOR;
    DoIO( (struct IORequest *) dd_io_req );

    return 0;
}

static int readDiskData( unsigned char *buffer,
                         int maxsize,
                         struct disk_buffer *disk_buf )
{
    int current_buffer_offset = 0;
    int remaining_bytes = maxsize;
    int bytes2copy;

    for (;;) {
        /* printf( "getting data from %d,%d\n",
           disk_buf->active_buffer,
           disk_buf->chip_buffer_offset[disk_buf->active_buffer] );
        */

        if ( disk_buf->chip_buffer_offset[disk_buf->active_buffer] < DISK_IO_BUFFER_SIZE ) {
            /* some bytes left in active buffer, copy them */

            bytes2copy = DISK_IO_BUFFER_SIZE - disk_buf->chip_buffer_offset[disk_buf->active_buffer];
            if ( bytes2copy > remaining_bytes ) {
                bytes2copy = remaining_bytes;
            }

            memcpy( buffer + current_buffer_offset,
                    disk_buf->chip_buffer[disk_buf->active_buffer] + disk_buf->chip_buffer_offset[disk_buf->active_buffer],
                    bytes2copy );

            disk_buf->chip_buffer_offset[disk_buf->active_buffer] += bytes2copy;
            remaining_bytes -= bytes2copy;
            current_buffer_offset += bytes2copy;
            disk_buf->total_read_bytes += bytes2copy;
        }

        /* printf( "Copied %d bytes from active buffer (%d remaining)\n",
           current_buffer_offset,
           DISK_IO_BUFFER_SIZE - disk_buf->chip_buffer_offset[disk_buf->active_buffer] );
        */

        if ( remaining_bytes == 0 ||
             disk_buf->total_read_bytes == disk_buf->total_size ) {
            return current_buffer_offset;
        }

        /* printf( "Waiting for IO\n" ); */

        if ( io_pending ) {
            io_pending = 0;
            if ( WaitIO( (struct IORequest *) dd_io_req ) != 0 ) {
                return -1;
            }
        }

        // issue next io into active buffer

        if ( disk_buf->current_io_offset < disk_buf->max_io_offset ) {
            /* printf( "Issue next io at %d\n",
               disk_buf->current_io_offset );
            */

            dd_io_req->io_Command = CMD_READ;
            dd_io_req->io_Length = DISK_IO_BUFFER_SIZE;
            dd_io_req->io_Data = disk_buf->chip_buffer[disk_buf->active_buffer];
            dd_io_req->io_Offset = disk_buf->current_io_offset;

            disk_buf->current_io_offset += DISK_IO_BUFFER_SIZE;

            SendIO( (struct IORequest *) dd_io_req );
            io_pending = 1;
        }

        /* printf( "Switch buffer\n" ); */

        // switch buffer
        disk_buf->active_buffer = ( disk_buf->active_buffer == 0 ) ? 1 : 0;
        disk_buf->chip_buffer_offset[disk_buf->active_buffer] = 0;
    }

    return 0;
}

int sendfile(const char *name, int mode)
{
    int sendfile;
    int bytecyc;
    int outchecksum,inchecksum,checksum,wrong,tc;
    int sendbytes,ts;
    double d1;
    ULONG secss,secso, tsecs;
    long secs;
    ULONG usecss,usecso, tusecs;
    long usecs;
    int count;
    char *outbuf, *sendtempl;
    char *outbasename;
    apc_command_t com;
    int ret=0;
    char buf4long[ BYTESFORNUMBER( long ) ];  // dieser Platz reicht fr eine long-Zahl mit Vorzeichen und NullByte
    struct disk_buffer disk_buf = { { NULL, NULL }, { 0, 0 }, 0, 0, 0, 0, 0 };
    int update_rate = 19; // default for parallel
    int rem, l;

    if ( config.mode == APCCOMM_SERIAL ) {
        update_rate = 5;
    }

    if(init_outgoing_file(name)!=0) {
#ifdef GUI
        rtEZRequest( "Can't open file %s!", "Okay", NULL, (struct TagItem *)rttags, name );
#else
        printf("can't open file %s!\n",name);
#endif
        return 1;
    }

    sendfile = 1;

    if ( adf_mode ) {
        outbasename = outfile.name;

        disk_buf.chip_buffer[0] = AllocMem( DISK_IO_BUFFER_SIZE, MEMF_CHIP );
        if ( ! disk_buf.chip_buffer[0] ) {
            ret = -1;
            sendfile = 0;
        }
        disk_buf.chip_buffer[1] = AllocMem( DISK_IO_BUFFER_SIZE, MEMF_CHIP );
        if ( ! disk_buf.chip_buffer[1] ) {
            FreeMem( disk_buf.chip_buffer[0], DISK_IO_BUFFER_SIZE );
            disk_buf.chip_buffer[0] = NULL;
            ret = -1;
            sendfile = 0;
        }

        disk_buf.total_size = outfile.size;
        disk_buf.current_io_offset = min_track * 512 * 11 * 2;
        disk_buf.max_io_offset = ( max_track + 1 ) * 512 * 11 * 2;

        if ( sendfile ) {
            dd_io_req->io_Command = CMD_READ;
            dd_io_req->io_Length = DISK_IO_BUFFER_SIZE;
            dd_io_req->io_Data = disk_buf.chip_buffer[0];
            dd_io_req->io_Offset = disk_buf.current_io_offset;

            disk_buf.current_io_offset += DISK_IO_BUFFER_SIZE;

            DoIO( (struct IORequest *) dd_io_req );

            dd_io_req->io_Command = CMD_READ;
            dd_io_req->io_Length = DISK_IO_BUFFER_SIZE;
            dd_io_req->io_Data = disk_buf.chip_buffer[1];
            dd_io_req->io_Offset = disk_buf.current_io_offset;

            disk_buf.current_io_offset += DISK_IO_BUFFER_SIZE;

            SendIO( (struct IORequest *) dd_io_req );

            io_pending = 1;
        }
    } else {
        if ( osversion >= 36 ) {
            outbasename=(char*)FilePart((char*)name);
        } else {
            outbasename = strrchr( name, '/' );
            if ( outbasename != NULL ) outbasename++;
            else {
                /* slash not found
                   try :
                */
                outbasename = strrchr( name, ':' );
                if ( outbasename != NULL ) outbasename++;
                else outbasename = (char*)name;
            }
        }
    }

#ifdef GUI
    GT_SetGadgetAttrs( Send_filesGadgets[ GD_sft ], Send_filesWnd, NULL, GTTX_Text, outbasename, TAG_END );
#else
    if ( ! BE_QUIET ) printf("sending file \"%s\", size %d\n",name,outfile.size);
#endif
    checksum=0;
    bytecyc=0;
    outputInt( 0, APC_COMMAND_FILETRANSFER );

    outputInt( 4, outfile.size );
    outputInt( 8, permAmiga2PC( mode ) );

    rem = maxOutputSize() - 16;
    l = min( strlen( outbasename ),
             rem );
    outputInt( 12, l );
    strncpy( outputData( 16 ),outbasename,l);
    setOutputSize( l + 16 );
    
    /* sending the command to the other side
     * it get received via recvcommand
     */
    am_transferblock();

    com = (apc_command_t)inputInt( 0 );
    if ( com != APC_COMMAND_ABORT ) {

        /* preparing progress display */
        sprintf( buf4long, "%d", outfile.size );
#ifdef GUI
#define SEND_PRE ""
#else
#define SEND_PRE "SEND: "
#endif
#define SEND_TEMPL "%%%dd/%%%dd (%%3d%%%%)"
        sendtempl = (char*)malloc( strlen( SEND_PRE ) +
                                   strlen( SEND_TEMPL ) +
                                   ( 2 * BYTESFORNUMBER( long ) ) + 1 );
        sprintf( sendtempl, SEND_PRE SEND_TEMPL, (int)strlen( buf4long ), (int)strlen( buf4long ) );
#undef SEND_PRE
#undef SEND_TEMPL
        outbuf = (char*)malloc( strlen( sendtempl ) +
                                ( 3 * BYTESFORNUMBER( long ) ) + 1 );
    
        inchecksum=outchecksum=0;
        sendbytes=0;
        wrong=0;
    
        /* now synchronize with other side and check state */
        if(recvcommand(&com,NULL,NULL)==0) {
            if(com==APC_COMMAND_NOP) {
                CurrentTime(&secss,&usecss);
                tsecs = secss;
                tusecs = usecss;
                for(count=0;sendfile==1;count++) {

                    if ( adf_mode ) {
                        ts = readDiskData( outputData( 0 ),
                                           maxOutputSize() - 4,
                                           &disk_buf );

                        // just for checksum calculation
                        readtobuffer( outputData( 0 ), ts, &outchecksum );
                    } else {
                        ts=readtobuffer(outputData( 0 ), maxOutputSize() - 4,&outchecksum);
                    }

                    sendbytes+=ts;
                    if ( ts < 1 ) sendfile=2;
                    if ( ts < 0 ) wrong = 3;

                    setOutputSize( ts + 4 );
                    if ( send_cancel == 1 ) {
                        outputTrailingInt( 0xffffffee );
                        wrong = 4;
                    } else if(wrong==0) outputTrailingInt( outchecksum );
                    else outputTrailingInt( 0xffffffff );

                    if ( ( sendfile == 1 ) || ( wrong != 0 ) ) {
                        if ( am_transferblock() != 0 ) wrong = 5;
                        tc = inputTrailingInt();
                        if(tc==0xffffffff) wrong=2;
                        else if ( tc == 0xffffffee ) wrong = 4;  /* abort */
                    }
                    if(wrong!=0) break;
	  
                    if ( ( count == update_rate ) || ( sendfile != 1 ) ) {
                        count=0;
                        if ( outfile.size )
                            sprintf( outbuf, sendtempl, sendbytes, outfile.size, ( sendbytes * 100 ) / outfile.size );
                        else
                            sprintf( outbuf, sendtempl, 0, 0, 100 );	      
#ifdef GUI
                        GT_SetGadgetAttrs( Send_filesGadgets[ GD_sbt ], Send_filesWnd, NULL, GTTX_Text, outbuf, TAG_END );
                        HandleSend_filesIDCMP();
#else
                        if ( ! BE_QUIET ) printf("\r%s",outbuf);
                        fflush(stdout);
#endif

#ifdef GUI
                        CurrentTime(&secso,&usecso);

                        secs=secso-tsecs;
                        usecs=usecso-tusecs;
                        if(usecs<0) {
                            secs--;
                            usecs+=1000000;
                        }
	  
                        if ( ( secs == 0 ) && ( usecs == 0 ) ) d1 = 0.0;
                        else {
                            d1 = ( update_rate + 1 ) * ( maxOutputSize() - 4 ); /* update_rate+1 blocks with each TRANSFERBS - 4 */
                            d1*=1000000;
                            d1/=(secs*1000000.0+usecs);
                            d1/=1024;
                        }
                        sprintf( buf4long, "%ld", (long)d1 );
                        GT_SetGadgetAttrs( Send_filesGadgets[ GD_srt ], Send_filesWnd, NULL, GTTX_Text, buf4long, TAG_END );
                        tsecs = secso;
                        tusecs = usecso;
#endif
                    }
                }

                if ( io_pending ) {
                    io_pending = 0;
                    if ( WaitIO( (struct IORequest *) dd_io_req ) != 0 ) {
                        /* IO will not be pending when finished regularily, only
                           in case of abort I need to wait for IO, but then I don't
                           care about status */
                    }
                }

                CurrentTime(&secso,&usecso);
#ifndef GUI
                if ( BE_VERBOSE ) printf("\nTransfer finished");
                else if ( ! BE_QUIET ) printf("\n");
#endif
                if(wrong==1) {
#ifdef GUI
                    rtEZRequest( "Error while transfer\nI have received wrong data!\n"
                                 "Please repeat transmission!\n"
                                 "Wrong file was %s", "Okay", NULL, (struct TagItem *)rttags, name );
#else
                    if ( BE_VERBOSE ) printf("\n");
                    printf("Error while transfer\nI have received wrong data!\n");
                    printf("Please repeat transmission!\n");
                    printf( "Wrong file was %s\n", name );
#endif
                } else if(wrong==2) {
#ifdef GUI
                    rtEZRequest( "Error while transfer\nThe PC reports data loss!\n"
                                 "Please repeat transmission!\n"
                                 "Wrong file was %s", "Okay", NULL, (struct TagItem *)rttags, name );
#else
                    if ( BE_VERBOSE ) printf("\n");
                    printf("Error while transfer\nThe PC reports data loss!\n");
                    printf("Please repeat transmission!\n");
                    printf( "Wrong file was %s\n", name );
#endif
                } else if(wrong==3) {
#ifdef GUI
                    rtEZRequest( "Error while reading file\n"
                                 "Wrong file was %s", "Okay", NULL, (struct TagItem *)rttags, name );
#else
                    if ( BE_VERBOSE ) printf("\n");
                    printf("Error while reading file\n");
                    printf( "Wrong file was %s\n", name );
#endif
                } else if ( wrong == 5 ) {
#ifdef GUI
                    rtEZRequest( "Error while transmitting data\n"
                                 "Wrong file was %s\nError was: %s", "Okay", NULL, (struct TagItem *)rttags, name, error_message );
#else
                    if ( BE_VERBOSE ) printf("\n");
                    printf("Error while transmitting data\n");
                    printf( "Wrong file was %s\n", name );
#endif
                } else if ( wrong == 4) {
                    /* no requester!, this is done by calling functions */
                    if ( BE_VERBOSE ) printf("\n");
                    ret = -1;
                } else {
                    secs=secso-secss;
                    usecs=usecso-usecss;
                    if(usecs<0) {
                        secs--;
                        usecs+=1000000;
                    }
	  
                    if ( ( secs == 0 ) && ( usecs == 0 ) ) d1 = 0.0;
                    else {
                        d1=outfile.size;
                        d1*=1000000;
                        d1/=(secs*1000000.0+usecs);
                        d1/=1024;
                    }
#ifdef GUI
                    sprintf( buf4long, "%ld", (long)d1 );
                    GT_SetGadgetAttrs( Send_filesGadgets[ GD_srt ], Send_filesWnd, NULL, GTTX_Text, buf4long, TAG_END );
#else
                    if ( BE_VERBOSE )
                        printf(", file sent with %d KBytes/s\n", (int)d1 );
#endif

                    tfl.files++;
                    tfl.bytes += outfile.size;
                    tfl.secs += secs;
                    tfl.usecs += usecs;
                    if ( tfl.usecs >= 1000000 ) {
                        tfl.secs++;
                        tfl.usecs -= 1000000;
                    }
                }
            } else {
                if(com==APC_COMMAND_SKIPFILE) {
#ifndef GUI
                    if ( ! BE_QUIET ) printf("Skipping file!\n");
#endif
                } else if(com==APC_COMMAND_ABORT) {
                    ret=-1;
                } else {
#ifdef GUI
                    rtEZRequest( "Wrong command %ld", "Okay", NULL, (struct TagItem *)rttags, com );
#else
                    printf("Wrong command %d\n",com);
#endif
                    ret=1;
                }
            }
        } else {
#ifdef GUI
            rtEZRequest( "Error while receiving command!", "Okay", NULL, (struct TagItem *)rttags );
#else
            printf("Error while receiving command!\n");
#endif
            ret=1;
        }

        if ( adf_mode ) {
            finalizeDisk();
        }

        free( sendtempl );
        free( outbuf );
    } else {
        ret = -1;
    }
    close_outgoing_file();
    free(outfile.name);

    if ( disk_buf.chip_buffer[0] ) {
        FreeMem( disk_buf.chip_buffer[0], DISK_IO_BUFFER_SIZE );
    }
    if ( disk_buf.chip_buffer[1] ) {
        FreeMem( disk_buf.chip_buffer[1], DISK_IO_BUFFER_SIZE );
    }

    return ret;
}

int sendcommand(apc_command_t com,const char *arg, int mode)
{
  if(com==APC_COMMAND_FILETRANSFER) {
    return sendfile(arg, mode);
  } else if(com==APC_COMMAND_ERROR) {
    outputInt( 0, com );
    setOutputSize( 4 );
    if ( am_transferblock() != 0 ) {
#ifdef GUI
        rtEZRequest( "Transmission error!", "Okay", NULL, (struct TagItem *)rttags );
#else
        fprintf( stderr, "Transmission error!\n" );
#endif
        return 1;
    }
  } else if(com==APC_COMMAND_ENTERDIR) {
      int rem = maxOutputSize();
      if( strlen( arg ) + 4 + 4 + 4 <= rem ) {
          outputInt( 0, com );
          outputInt( 4, permAmiga2PC( mode ) );

          int l = min( rem - 12,
                       strlen( arg ) );
          outputInt( 8, l );
          strncpy( (char*)( outputData( 12 ) ), arg, l );
          setOutputSize( l + 12 );
          if ( am_transferblock() != 0 ) {
#ifdef GUI
              rtEZRequest( "Transmission error!", "Okay", NULL, (struct TagItem *)rttags );
#else
              fprintf( stderr, "Transmission error!\n" );
#endif
              return 1;
          }
      } else {
#ifdef GUI
          rtEZRequest( "Too long dirname!", "Okay", NULL, (struct TagItem *)rttags );
#else
          printf("Too long dirname!\n");
#endif
          return 1;
      }
  } else if(com==APC_COMMAND_LEAVEDIR) {
    outputInt( 0, com );
    setOutputSize( 4 );
    if ( am_transferblock() != 0 ) {
#ifdef GUI
        rtEZRequest( "Transmission error!", "Okay", NULL, (struct TagItem *)rttags );
#else
        fprintf( stderr, "Transmission error!\n" );
#endif
        return 1;
    }
  } else if(com==APC_COMMAND_FINISHED) {
    outputInt( 0, com );
    setOutputSize( 4 );
    if ( am_transferblock() != 0 ) {
#ifdef GUI
        rtEZRequest( "Transmission error!", "Okay", NULL, (struct TagItem *)rttags );
#else
        fprintf( stderr, "Transmission error!\n" );
#endif
        return 1;
    }
  } else if(com==APC_COMMAND_NOP) {
    outputInt( 0, com );
    setOutputSize( 4 );
    if ( am_transferblock() != 0 ) {
#ifdef GUI
        rtEZRequest( "Transmission error!", "Okay", NULL, (struct TagItem *)rttags );
#else
        fprintf( stderr, "Transmission error!\n" );
#endif
        return 1;
    }
  } else if(com==APC_COMMAND_ABORT) {
    outputInt( 0, com );
    setOutputSize( 4 );
    if ( am_transferblock() != 0 ) {
#ifdef GUI
        rtEZRequest( "Transmission error!", "Okay", NULL, (struct TagItem *)rttags );
#else
        fprintf( stderr, "Transmission error!\n" );
#endif
        return 1;
    }
  } else if(com==APC_COMMAND_SKIPFILE) {
    outputInt( 0, com );
    setOutputSize( 4 );
    if ( am_transferblock() != 0 ) {
#ifdef GUI
        rtEZRequest( "Transmission error!", "Okay", NULL, (struct TagItem *)rttags );
#else
        fprintf( stderr, "Transmission error!\n" );
#endif
        return 1;
    }
  } else {
#ifdef GUI
    rtEZRequest( "Unknown command %ld to send!", "Okay", NULL, (struct TagItem *)rttags, com );
#else
    printf("Unknown command %d to send!\n",com);
#endif
    return 1;
  }
  return 0;
}

int recvcommand(apc_command_t *com,char **arg, int *mode)
{
  apc_command_t tc;
  char *narg;

  setOutputSize( 0 );
  if ( am_transferblock() != 0 ) {
#ifdef GUI
      rtEZRequest( "Transmission error!", "Okay", NULL, (struct TagItem *)rttags );
#else
      fprintf( stderr, "Transmission error!\n" );
#endif
      return 1;
  }
  tc=(apc_command_t)inputInt( 0 );
  switch(tc) {
    case APC_COMMAND_ERROR:
      *com=tc;
      if(arg!=NULL) *arg=NULL;
      if(mode!=NULL) *mode=0;
      break;
    case APC_COMMAND_ENTERDIR:
        if(arg!=NULL) {
            int l;
            int rem;

            *mode = permPC2Amiga( inputInt( 4 ) );
            l = inputInt( 8 );
            rem = inputSize() - 12;
            if ( l > rem ) l = rem;
            narg=(char*)malloc(l+1);
            if(narg!=NULL) {
                *com=tc;
                const unsigned char *data = inputData( 12 );
                strncpy( narg, (char*)( data ), l );
                narg[l]='\0';
                *arg=narg;
            } else return 1;
        } else return 1;
        break;
    case APC_COMMAND_LEAVEDIR:
      *com=tc;
      if(arg!=NULL) *arg=NULL;
      if(mode!=NULL) *mode=0;
      break;
    case APC_COMMAND_FILETRANSFER:
      *com=tc;
      if(arg!=NULL) *arg=NULL;
      if(mode!=NULL) *mode=0;
      return recvfile();
      break;
    case APC_COMMAND_FINISHED:
      *com=tc;
      if(arg!=NULL) *arg=NULL;
      if(mode!=NULL) *mode=0;
      break;
    case APC_COMMAND_NOP:
      *com=tc;
      if(arg!=NULL) *arg=NULL;
      if(mode!=NULL) *mode=0;
      break;
    case APC_COMMAND_ABORT:
      *com=tc;
      if(arg!=NULL) *arg=NULL;
      if(mode!=NULL) *mode=0;
      break;
    case APC_COMMAND_SKIPFILE:
      *com=tc;
      if(arg!=NULL) *arg=NULL;
      if(mode!=NULL) *mode=0;
      break;
    default:
#ifdef GUI
      rtEZRequest( "Unknown command %ld received!", "Okay", NULL, (struct TagItem *)rttags, tc );
#else
      printf("Unknown command %d received!\n",tc);
#endif
      return 1;
      break;
  }
  return 0;
}

static int writeBufferToDisk( UBYTE *buffer,
                              int byte_offset )
{
    if ( io_pending ) return 1;

    if ( ( byte_offset % DISK_IO_BUFFER_SIZE ) != 0 ) return 1;

    dd_io_req->io_Command = CMD_WRITE;
    dd_io_req->io_Length = DISK_IO_BUFFER_SIZE;
	dd_io_req->io_Data = buffer;
    dd_io_req->io_Offset = byte_offset;

    if ( config.mode == APCCOMM_SERIAL ) {
        // serial mode is interupt critical so do not write data
        // asynchronously
        if ( DoIO( (struct IORequest *) dd_io_req ) != 0 ) return 1;
    } else {
        SendIO( (struct IORequest *) dd_io_req );
        io_pending = 1;
    }

    return 0;
}

static int verifyBufferWithDisk( UBYTE *buffer,
                                 int byte_offset,
                                 UBYTE *compare_buffer )
{
    if ( io_pending ) return 1;

    if ( ( byte_offset % DISK_IO_BUFFER_SIZE ) != 0 ) return 1;

    dd_io_req->io_Command = CMD_READ;
    dd_io_req->io_Length = DISK_IO_BUFFER_SIZE;
    dd_io_req->io_Data = buffer;
    dd_io_req->io_Offset = byte_offset;

    if ( DoIO( (struct IORequest *) dd_io_req ) != 0 ) return 1;

    if ( memcmp( buffer, compare_buffer, DISK_IO_BUFFER_SIZE ) != 0 ) return 1;

    return 0;
}

static int verifyDisk( UBYTE *buffer,
                       int total_checksum,
                       int size,
                       char *outbuf,
                       char *recvtempl )
{
    int p = 0;
    int i, t;
    int new_total_checksum = 0;
    int tchk = 0;
    int total_pos = 0;

    while ( p < size ) {
        dd_io_req->io_Command = CMD_READ;
        dd_io_req->io_Length = DISK_IO_BUFFER_SIZE;
        dd_io_req->io_Data = buffer;
        dd_io_req->io_Offset = p;

        p += DISK_IO_BUFFER_SIZE;

        DoIO( (struct IORequest *) dd_io_req );

        if ( p ) {
            sprintf( outbuf, recvtempl, p, size, ( p * 100 ) / size );
        } else {
            sprintf( outbuf, recvtempl, 0, 0, 100 );
        }
#ifdef GUI
        GT_SetGadgetAttrs( Receive_filesGadgets[ GD_rbt ], Receive_filesWnd, NULL, GTTX_Text, outbuf, TAG_END );
        HandleReceive_filesIDCMP();
#else
        if ( ! BE_QUIET ) printf("\r%s",outbuf);
        fflush( stdout );
#endif

        for ( i = 0; i < DISK_IO_BUFFER_SIZE; i++, total_pos++ ) {
            tchk += buffer[i];

            if ( ( total_pos > 0 &&
                   ( ( total_pos + 1 ) % ( maxInputSize() - 4 ) ) == 0 ) ||
                 total_pos + 1 == size ) {
                t = new_total_checksum >> 24;
                new_total_checksum <<= 8;
                new_total_checksum |= t;

                new_total_checksum += tchk;

                tchk = 0;
            }
        }
    }

    if ( total_checksum == new_total_checksum ) return 0;

    return 1;
}

#define DIRECT_VERIFY

int recvfile( void )
{
    char *inarg=NULL;
    int s1;
    int getfile;
    int bytecyc;
    int outchecksum,inchecksum,checksum,wrong,tc;
    int size,recbytes;
    double d1;
    ULONG secss,secsi;
#ifdef GUI
    ULONG tsecs, tusecs;
#endif
    long secs;
    ULONG usecss,usecsi;
    long usecs;
    int count;
    char *outbuf, *recvtempl,*cwd,*tstr=NULL;
    int cwdsize;
    int l;
    int ret;
    int mode;
    char buf4long[ 3 * 4 + 1 + 1 ];  // dieser Platz reicht fr eine 4 Byte-Zahl mit Vorzeichen und NullByte
    UBYTE *chip_buffer[2] = { NULL, NULL };
    UBYTE *verify_buffer = NULL;
    short chip_buffer_offset[2] = { 0, 0 };
    short active_buffer = 0;
    int current_sector = 0;
    int total_checksum = 0;
    int update_rate = 19; // default for parallel
    int rem;
    const unsigned char *data;

    if ( config.mode == APCCOMM_SERIAL ) {
        update_rate = 5;
    }

    size = inputInt( 4 );
    mode = inputInt( 8 );
    l = inputInt( 12 );

    rem = inputSize() - 16;

    inarg=(char*)malloc(l+1);
    if(inarg==NULL) return 1;
    data = inputData( 16 );
    strncpy( inarg, (char*)( data ), min( l, rem ) );
    inarg[l]='\0';

    mode = permPC2Amiga( mode );
  
#ifdef GUI
    GT_SetGadgetAttrs( Receive_filesGadgets[ GD_rft ], Receive_filesWnd, NULL, GTTX_Text, inarg, TAG_END );
#else
    if ( ! BE_QUIET ) printf("receiving file \"%s\", size %d\n",inarg,size);
#endif
    ret = init_incoming_file( inarg, size );
    getfile=1;

    checksum=0;
    bytecyc=0;

    /* preparing progress display */
    sprintf( buf4long, "%d", size );
#ifdef GUI
#define RECV_PRE ""
#else
#define RECV_PRE "RECV: "
#endif
#define RECV_TEMPL "%%%dd/%%%dd (%%3d%%%%)"
    recvtempl = (char*)malloc( strlen( RECV_PRE ) +
                               strlen( RECV_TEMPL ) +
                               ( 2 * BYTESFORNUMBER( long ) ) + 1 );
    sprintf( recvtempl, RECV_PRE RECV_TEMPL, (int)strlen( buf4long ), (int)strlen( buf4long ) );
#undef RECV_PRE
#undef RECV_TEMPL
    outbuf = (char*)malloc( strlen( recvtempl ) +
                            ( 3 * BYTESFORNUMBER( long ) ) + 1 );

#ifndef GUI
    if ( BE_VERBOSE ) printf("Starting transfer\n");
#endif
    inchecksum=outchecksum=0;
    recbytes=0;
    wrong=0;
    tc = 0;

    if ( adf_mode ) {
        chip_buffer[0] = AllocMem( DISK_IO_BUFFER_SIZE, MEMF_CHIP );
        if ( ! chip_buffer[0] ) {
            ret = -1;
        }
        chip_buffer[1] = AllocMem( DISK_IO_BUFFER_SIZE, MEMF_CHIP );
        if ( ! chip_buffer[1] ) {
            FreeMem( chip_buffer[0], DISK_IO_BUFFER_SIZE );
            chip_buffer[0] = NULL;
            ret = -1;
        }

        verify_buffer = AllocMem( DISK_IO_BUFFER_SIZE, MEMF_CHIP );
        if ( ! verify_buffer ) {
            FreeMem( chip_buffer[0], DISK_IO_BUFFER_SIZE );
            chip_buffer[0] = NULL;
            FreeMem( chip_buffer[1], DISK_IO_BUFFER_SIZE );
            chip_buffer[1] = NULL;
            ret = -1;
        }
    }

    if ( ret > 0 ) {
#ifdef GUI
        rtEZRequest( "Can't create %s in %s, aborting", "Okay", NULL, (struct TagItem *)rttags, inarg, basedir );
#else
        printf( "Can't create %s in %s, aborting\n", inarg, basedir );
#endif
        sendcommand(APC_COMMAND_ABORT,NULL,0);
    } else if ( ret == -2 ) {
#ifndef GUI
        printf( "Aborting\n" );
#endif
        sendcommand( APC_COMMAND_ABORT, NULL, 0 );
    } else if(ret==-1) {
        sendcommand(APC_COMMAND_SKIPFILE,NULL,0);
        ret = 0;
    } else if(ret==0) {
        sendcommand(APC_COMMAND_NOP,NULL,0);

        CurrentTime(&secss,&usecss);
        for(count=0;getfile==1;count++) {
            if(recbytes<size) {
                setOutputSize( 4 );
                if (receive_cancel == 1) {
                    outputTrailingInt( 0xffffffee );
                    wrong = 4;
                } else if(wrong==0) outputTrailingInt( 0 );
                else outputTrailingInt( 0xffffffff );

                if(getfile==1) {
                    if ( am_transferblock() != 0 ) wrong = 5;
                    tc = inputTrailingInt();
                    if(tc==0xffffffff) wrong=2;
                    else if ( tc == 0xffffffee ) wrong = 4;
                }
                if(wrong!=0) break;
	
                rem = inputSize();

                s1=size-recbytes;
                if(s1>rem - 4) s1=rem - 4;
                else getfile=2;

                if ( adf_mode ) {
                    /* I need to copy data into chipmem buffer also since
                       I transfer 508 bytes with each block and not 512 */
                    int remaining_bytes = s1;
                    int inblock_offset = 0;

                    while ( remaining_bytes > 0 ) {
                        short required_buffer_bytes = DISK_IO_BUFFER_SIZE - chip_buffer_offset[active_buffer];

                        if ( remaining_bytes < required_buffer_bytes ) {
                            required_buffer_bytes = remaining_bytes;
                        }

                        memcpy( chip_buffer[active_buffer] + chip_buffer_offset[active_buffer],
                                inputData( 0 ) + inblock_offset,
                                required_buffer_bytes );

                        chip_buffer_offset[active_buffer] += required_buffer_bytes;
                        inblock_offset += required_buffer_bytes;
                        remaining_bytes -= required_buffer_bytes;

                        if ( chip_buffer_offset[active_buffer] == DISK_IO_BUFFER_SIZE ) {
                            if ( io_pending ) {
                                if ( WaitIO( (struct IORequest *) dd_io_req ) != 0 ) {
                                    wrong = 3;
                                }
                                io_pending = 0;

#ifdef DIRECT_VERIFY
                                if ( adf_write_verify &&
                                     verifyBufferWithDisk( verify_buffer,
                                                           (current_sector - 1) * DISK_IO_BUFFER_SIZE,
                                                           chip_buffer[1 - active_buffer] ) != 0 ) {
                                    wrong = 3;
                                }
#endif
                            }

                            if ( writeBufferToDisk( chip_buffer[active_buffer],
                                                    (current_sector++) * DISK_IO_BUFFER_SIZE ) != 0 ) {
                                wrong = 3;
                            }
                            chip_buffer_offset[active_buffer] = 0;
                            active_buffer = ( active_buffer == 0 ) ? 1 : 0;
                        }
                    }
                }

                /* this will just calculate the checksum in case of adf_mode */
                writefrombuffer(inputData( 0 ),s1,&inchecksum);

                recbytes+=s1;
                checksum=inchecksum-tc;
                if(checksum!=0) wrong=1;

                if ( adf_mode ) {
                    int t = total_checksum >> 24;
                    total_checksum <<= 8;
                    total_checksum |= t;

                    total_checksum += inchecksum;
                }
            } else getfile = 2;
      
            if ( ( count == update_rate ) || ( getfile != 1 ) ) {
                count=0;
                if ( size )
                    sprintf( outbuf, recvtempl, recbytes, size, ( recbytes * 100 ) / size );
                else
                    sprintf( outbuf, recvtempl, 0, 0, 100 );
#ifdef GUI
                GT_SetGadgetAttrs( Receive_filesGadgets[ GD_rbt ], Receive_filesWnd, NULL, GTTX_Text, outbuf, TAG_END );
                HandleReceive_filesIDCMP();
#else
                if ( ! BE_QUIET ) printf("\r%s",outbuf);
                fflush(stdout);
#endif

#ifdef GUI
                CurrentTime(&secsi,&usecsi);

                secs=secsi-tsecs;
                usecs=usecsi-tusecs;
                if(usecs<0) {
                    secs--;
                    usecs+=1000000;
                }
	  
                if ( ( secs == 0 ) && ( usecs == 0 ) ) d1 = 0.0;
                else {
                    d1 = ( update_rate + 1 ) * ( maxInputSize() - 4 ); /* update_rate+1 blocks with each TRANSFERBS/2 - 4 */
                    d1*=1000000;
                    d1/=(secs*1000000.0+usecs);
                    d1/=1024;
                }
                sprintf( buf4long, "%ld", (long)d1 );
                GT_SetGadgetAttrs( Receive_filesGadgets[ GD_rrt ], Receive_filesWnd, NULL, GTTX_Text, buf4long, TAG_END );
                tsecs = secsi;
                tusecs = usecsi;
#endif

            }
        }

        if ( adf_mode && io_pending ) {
            if ( WaitIO( (struct IORequest *) dd_io_req ) != 0 ) {
                wrong = 3;
            }
            io_pending = 0;

#ifdef DIRECT_VERIFY
            if ( adf_write_verify &&
                 verifyBufferWithDisk( verify_buffer,
                                       (current_sector - 1) * DISK_IO_BUFFER_SIZE,
                                       chip_buffer[1 - active_buffer] ) != 0 ) {
                wrong = 3;
            }
#endif
        }

        /* this works but is disabled in flavor of direct verifying */
#ifndef DIRECT_VERIFY
        if ( adf_mode && adf_write_verify ) {
#ifdef GUI
            GT_SetGadgetAttrs( Receive_filesGadgets[ GD_rft ], Receive_filesWnd, NULL, GTTX_Text, "Verify...", TAG_END );
            GT_SetGadgetAttrs( Receive_filesGadgets[ GD_rrt ], Receive_filesWnd, NULL, GTTX_Text, "", TAG_END );
#else
            if ( ! BE_QUIET ) printf( "verify...\n" );
#endif
            if ( verifyDisk( chip_buffer[0],
                             total_checksum,
                             size,
                             outbuf, recvtempl ) != 0 ) {
                wrong = 3;
            }
        }
#endif

        CurrentTime(&secsi,&usecsi);

#ifndef GUI
        if ( BE_VERBOSE ) printf("\nTransfer finished");
        else if ( ! BE_QUIET ) printf("\n");
#endif
        if(wrong==1) {
#ifdef GUI
            rtEZRequest( "Error while transfer\nI have received wrong data!\n"
                         "Please repeat transmission!", "Okay", NULL, (struct TagItem *)rttags );
#else
            if ( BE_VERBOSE ) printf("\n");
            printf("Error while transfer\nI have received wrong data!\n");
            printf("Please repeat transmission!\n");
#endif
        } else if(wrong==2) {
#ifdef GUI
            rtEZRequest( "Error while transfer\nThe PC reports data loss!\n"
                         "Please repeat transmission!", "Okay", NULL, (struct TagItem *)rttags );
#else
            if ( BE_VERBOSE ) printf("\n");
            printf("Error while transfer\nThe PC reports data loss!\n");
            printf("Please repeat transmission!\n");
#endif
        } else if ( wrong == 3 ) {
#ifdef GUI
            rtEZRequest( "Error while writing file\n"
                         "Wrong file was %s", "Okay", NULL, (struct TagItem *)rttags, inarg );
#else
            if ( BE_VERBOSE ) printf("\n");
            printf( "Error while writing file\n" );
            printf( "Wrong file was %s\n", inarg );
#endif
        } else if ( wrong == 5 ) {
#ifdef GUI
            rtEZRequest( "Error while transmittind data\n"
                         "Wrong file was %s\nError was: %s", "Okay", NULL, (struct TagItem *)rttags, inarg, error_message );
#else
            if ( BE_VERBOSE ) printf("\n");
            printf( "Error while transmitting data\n" );
            printf( "Wrong file was %s\n", inarg );
#endif
        } else if ( wrong == 4 ) {
            if ( BE_VERBOSE ) printf("\n");
            ret = -1;
        } else {
            secs=secsi-secss;
            usecs=usecsi-usecss;
            if(usecs<0) {
                secs--;
                usecs+=1000000;
            }
      
            if ( ( secs == 0 ) && ( usecs == 0 ) ) d1 = 0.0;
            else {
                d1=size;
                d1*=1000000;
                d1/=(secs*1000000.0+usecs);
                d1/=1024;
            }
#ifdef GUI
            sprintf( buf4long, "%ld", (long)d1 );
            GT_SetGadgetAttrs( Receive_filesGadgets[ GD_rrt ], Receive_filesWnd, NULL, GTTX_Text, buf4long, TAG_END );
#else
            if ( BE_VERBOSE )
                printf(", file received with %d KBytes/s\n", (int)d1 );
#endif

            tfl.files++;
            tfl.bytes += size;
            tfl.secs += secs;
            tfl.usecs += usecs;
            if ( tfl.usecs >= 1000000 ) {
                tfl.secs++;
                tfl.usecs -= 1000000;
            }
        }

        if ( adf_mode ) {
            finalizeDisk();
        }

        close_incoming_file();
    
        /* File is stored in the current dir when basedir is NULL
         * File is stored relative to current dir in basedir when basedir contains NOT a ":"
         * File is stored in basedir when basedir contains a ":"
         */

        /* First get the cwd */
        cwdsize = 1024;
        cwd = (char*) malloc( cwdsize );
        if(cwd!=NULL) {
            while(getcwd(cwd,cwdsize)==NULL) {
                if(errno==ERANGE) {
                    free(cwd);
                    cwdsize*=2;
                    cwd=(char*)malloc(cwdsize);
                    if(cwd==NULL) break;
                } else {
                    free(cwd);
                    cwd=NULL;
                }
            }
        }

        if ( ! adf_mode ) {
            /* now check all 3 cases */

            if ( basedir == NULL ) {

                /* just in the cwd */

                if(cwd!=NULL) {
                    if(cwd[strlen(cwd)-1]==':') {
                        tstr=(char*)malloc(strlen(cwd)+strlen(infile.name)+1);
                        if(tstr!=NULL) sprintf(tstr,"%s%s",cwd,infile.name);
                    } else {
                        tstr=(char*)malloc(strlen(cwd)+1+strlen(infile.name)+1);
                        if(tstr!=NULL) sprintf(tstr,"%s/%s",cwd,infile.name);
                    }
                }
            } else {

                /* at least relative to basedir */

                if ( strrchr( basedir, ':' ) == NULL ) {

                    /* basedir is relative */

                    if ( cwd[strlen( cwd ) - 1] == ':' ) {
                        tstr = (char*) malloc( strlen( cwd ) + strlen( basedir ) + strlen( infile.name ) + 1 );
                        if ( tstr != NULL ) sprintf( tstr, "%s%s%s", cwd, basedir, infile.name );
                    } else {
                        tstr = (char*) malloc( strlen( cwd ) + 1 + strlen( basedir ) + strlen( infile.name ) + 1 );
                        if(tstr!=NULL) sprintf( tstr, "%s/%s%s", cwd, basedir, infile.name );
                    }
                } else {
                    tstr = (char*) malloc( strlen( basedir ) + strlen( infile.name ) + 1 );
                    if ( tstr != NULL ) sprintf( tstr, "%s%s", basedir, infile.name );
                }
            }

            if ( tstr != NULL ) {
#ifndef GUI
                if ( ( BE_VERBOSE ) || ( wrong ) ) printf( "Incoming file is stored at: %s\n", tstr );
#endif

                if ( ! ignore_prot )
                    if ( SetProtection( tstr, mode ) == 0 )
#ifdef GUI
                        rtEZRequest( "Can't restore file protection to %s", "Okay", NULL, (struct TagItem *)rttags, tstr );
#else
                printf( "Can't restore file protection to %s\n", tstr );
#endif

                free( tstr );
            }
        }

        if ( cwd != NULL) free(cwd);
        free( infile.name );
    }

    if ( chip_buffer[0] ) {
        FreeMem( chip_buffer[0], DISK_IO_BUFFER_SIZE );
    }
    if ( chip_buffer[1] ) {
        FreeMem( chip_buffer[1], DISK_IO_BUFFER_SIZE );
    }
    if ( verify_buffer ) {
        FreeMem( verify_buffer, DISK_IO_BUFFER_SIZE );
    }

    free( outbuf );
    free( recvtempl );
    free(inarg);
    return ret;
}
