/* apccomm_am_tr_lowlevel.c
   Part of "APCComm" 
   Copyright (C) 2022-2023 Ralf Hoffmann
   Contact: ralf@boomerangsworld.de

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
  
*/

#include "apccomm_all.h"
#include "apccomm_am_tr.h"
#include "apccomm_am.h"
#include "apccomm_am_main.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <sys/time.h>

#ifdef GUI
#include "apccomm_gui.h"
#endif

#include <clib/alib_protos.h>
#include <clib/exec_protos.h>
#include <devices/trackdisk.h>
#include <dos/dos.h>
#include <exec/devices.h>
#include <exec/exec.h>
#include <exec/io.h>
#include <exec/libraries.h>
#include <exec/memory.h>
#include <exec/types.h>
#include <proto/dos.h>
#include <proto/exec.h>
#include <proto/icon.h>
#include <proto/intuition.h>
#include <proto/misc.h>
#include <proto/wb.h>
#include <resources/misc.h>
#include <workbench/startup.h>
#include <workbench/workbench.h>
#include <devices/serial.h>

#define DDRA 0xbfd200
#define DDRB 0xbfe301
#define PRA 0xbfd000
#define PRB 0xbfe101
#define ICR 0xbfed01

#define MIN( a, b ) ((a) < (b) ? (a) : (b))

unsigned char *icr=(unsigned char*)ICR;
unsigned char *prb=(unsigned char*)PRB;
unsigned char *pra=(unsigned char*)PRA;
unsigned char *ddrb=(unsigned char*)DDRB;
unsigned char *ddra=(unsigned char*)DDRA;

int (*am_sync)( void ) = NULL;
int (*am_transferblock)( void ) = NULL;
int (*am_cleanup)( void ) = NULL;

int lastvalue;

struct Library *MiscBase = NULL;
short DevOpened = 0;

static struct MsgPort *SerialMP = NULL;
static struct IOExtSer *SerialIO = NULL;

int getack( void )
{
  unsigned char v;
  v=*pra;
  lastvalue=v&3;
  return ((v&4)==0)?0:1;
}

int getvalue( void )
{
  getack();
  return lastvalue;
}

void setack(int value)
{
  unsigned char v;
  v=*prb;
  if(value==0) v&=0xf;
  else v|=0x10;
  *prb=v;
}

void putnibble(int value)
{
  unsigned char v,nv,ack;
  v=*prb;
  ack=v&0x10;
  nv=value&0xf;
  nv|=ack;
  *prb=nv;
}

void putnibbleack(int value,int ack)
{
  unsigned char v;
  v=value&0xf;
  v|=(ack&0x1)<<4;
  *prb=v;
}

static int synchro( void )
{
  int count, pc = 0;
#ifdef GUI
  char buf[11];
  struct Gadget *text;
  
  text = ConnectWindowGadgets[ GD_ct ];
  buf[0] = '\0';
#endif

  setack(1);
  putnibble(0);
  for(count=0;getack()!=0;count++) {
    Delay(1);
#ifdef GUI
    HandleConnectWindowIDCMP();
#endif
    if((count%25)==0) {
      pc = ( pc + 1 ) % 10;
#ifdef GUI
      memset( buf, '.', pc );
      buf[pc] = '\0';
      GT_SetGadgetAttrs( text, ConnectWindowWnd, NULL, GTTX_Text, buf, TAG_END );
#else
      if ( ! BE_QUIET ) printf(".");
      fflush(stdout);
#endif
    }
    if ( connect_cancel == 1 ) return 1;
  }
  putnibble(1);
  for(count=0;getvalue()!=1;count++) {
    Delay(1);
#ifdef GUI
    HandleConnectWindowIDCMP();
#endif
    if((count%25)==0) {
      pc = ( pc + 1 ) % 10;
#ifdef GUI
      memset( buf, '.', pc );
      buf[pc] = '\0';
      GT_SetGadgetAttrs( text, ConnectWindowWnd, NULL, GTTX_Text, buf, TAG_END );
#else
      if ( ! BE_QUIET ) printf(".");
      fflush(stdout);
#endif
    }
    if ( connect_cancel == 1 ) return 1;
  }
  putnibble(2);
  for(count=0;getvalue()!=2;count++) {
    Delay(1);
#ifdef GUI
    HandleConnectWindowIDCMP();
#endif
    if((count%25)==0) {
      pc = ( pc + 1 ) % 10;
#ifdef GUI
      memset( buf, '.', pc );
      buf[pc] = '\0';
      GT_SetGadgetAttrs( text, ConnectWindowWnd, NULL, GTTX_Text, buf, TAG_END );
#else
      if ( ! BE_QUIET ) printf(".");
      fflush(stdout);
#endif
    }
    if ( connect_cancel == 1 ) return 1;
  }
  setack(1);

  return 0;
}

static int initParPort( void )
{
  MiscBase = (struct Library *)OpenResource( "misc.resource" );
  if ( MiscBase == NULL ) {
      sprintf( error_message, "Couldn't get misc.resource!" );
      return 1;
  }

  if ( AllocMiscResource( MR_PARALLELPORT,
                          "APCComm" ) ) {
      /* already allocated by another program */
      sprintf( error_message, "parallel port already allocated by another program!" );
      return 2;
  }

  DevOpened = 1;
  *ddrb=0xff;
  *ddra=0x00;
  return 0;
}

static int am_cleanup_par()
{
    if ( DevOpened != 0 ) {
      FreeMiscResource( MR_PARALLELPORT );
    }
    return 0;
}

static int ser_close( void );

static int ser_init( const struct apccomm_transfer_config *config )
{
    if ( osversion < 36 ) {
        SerialMP = CreatePort("apccomm",0);
    } else {
        SerialMP = CreateMsgPort();
    }
    if ( ! SerialMP ) {
        sprintf( error_message, "Failed to create msg port" );
        return -EINVAL;
    }

    if ( osversion < 36 ) {
        SerialIO = (struct IOExtSer*)CreateExtIO( SerialMP,
                                                  sizeof( *SerialIO ) );
    } else {
        SerialIO = CreateIORequest( SerialMP,
                                    sizeof( *SerialIO ) );
    }

    if ( ! SerialIO ) {
        sprintf( error_message, "Failed to create IO request" );
        ser_close();
        return -EINVAL;
    }

    UBYTE flags;
    // set 7wire (CRTSCTS)
    flags = SERF_7WIRE;
    flags &= ~SERF_PARTY_ON;
    flags |= SERF_XDISABLED;
    flags |= SERF_RAD_BOOGIE;

    SerialIO->io_SerFlags = flags;

    if ( OpenDevice( SERIALNAME, 0, (struct IORequest *)SerialIO, 0 ) != 0 ) {
        sprintf( error_message, "Failed to open serial device" );
        ser_close();
        return -EINVAL;
    }

    // use complete flags again, including 7wire. The bit will be
    // reset after OpenDevice and must be set again to take affect
    SerialIO->io_SerFlags = flags;

    if ( config->serial_speed == 9600 ) {
        SerialIO->io_Baud = 9600;
    } else if ( config->serial_speed == 19200 ) {
        SerialIO->io_Baud = 19200;
    } else if ( config->serial_speed == 38400 ) {
        SerialIO->io_Baud = 38400;
    } else if ( config->serial_speed == 115200 ) {
        SerialIO->io_Baud = 115200;
    } else {
        SerialIO->io_Baud = 57600;
    }

    if ( BE_VERBOSE ) {
#ifndef GUI
        printf( "Using serial speed %d\n", SerialIO->io_Baud );
#endif
    }

    SerialIO->io_RBufLen = 2048;

    SerialIO->io_ReadLen = SerialIO->io_WriteLen = 8;
    SerialIO->io_StopBits = 1;
    
    SerialIO->IOSer.io_Command  = SDCMD_SETPARAMS;
    if ( DoIO( (struct IORequest *)SerialIO ) != 0 ) {
        sprintf( error_message, "Failed to set parameters" );
        ser_close();
        return -EINVAL;
    }

    DevOpened = TRUE;

    return 0;
}

static void ser_drain( void )
{
    char buf[128];
    LONG res;

    SerialIO->IOSer.io_Command  = CMD_CLEAR;
    DoIO( (struct IORequest *)SerialIO );

    while ( TRUE ) {
        UWORD Serial_Status;
        ULONG Unread_Chars;

        SerialIO->IOSer.io_Command  = SDCMD_QUERY;
        DoIO( (struct IORequest *)SerialIO );

        Serial_Status = SerialIO->io_Status;
        Unread_Chars = SerialIO->IOSer.io_Actual;

        if ( Unread_Chars == 0 ) {
            break;
        }

        SerialIO->IOSer.io_Length   = MIN( Unread_Chars, sizeof( buf ) );
        SerialIO->IOSer.io_Data     = (APTR)&buf[0];
        SerialIO->IOSer.io_Command  = CMD_READ;
        res = DoIO((struct IORequest *)SerialIO);

        if ( res != 0 ) {
            if ( BE_VERBOSE ) {
#ifndef GUI
                printf( "Failed to drain: %d\n", res );
#endif
            }
            return;
        }
    }
}

static int ser_sync( void )
{
    char buf[16];
    LONG res;
    int count, pc = 0;
    int invalid_count = 0;
#ifdef GUI
    char progress_buf[11];
    struct Gadget *text;
  
    text = ConnectWindowGadgets[ GD_ct ];
    progress_buf[0] = '\0';
#endif

    ser_drain();

#ifndef GUI
    if ( BE_VERBOSE ) {
        printf( "Client start sync\n" );
    }
#endif

    while ( TRUE ) {
        SerialIO->IOSer.io_Length   = 1;
        SerialIO->IOSer.io_Data     = (APTR)&buf[0];
        SerialIO->IOSer.io_Command  = CMD_READ;
        SendIO((struct IORequest *)SerialIO);
        count++;

        while ( TRUE ) {
            Delay(1);

#ifdef GUI
            HandleConnectWindowIDCMP();
#endif
            if ( ( count % 25 ) == 0 ) {
                pc = ( pc + 1 ) % 10;
#ifdef GUI
                memset( progress_buf, '.', pc );
                progress_buf[pc] = '\0';
                GT_SetGadgetAttrs( text, ConnectWindowWnd, NULL, GTTX_Text, progress_buf, TAG_END );
#else
                if ( ! BE_QUIET ) printf(".");
                fflush(stdout);
#endif
            }

            if ( CheckIO( (struct IORequest *)SerialIO  ) != NULL ) {
                res = WaitIO( (struct IORequest *)SerialIO );
                break;
            }

            if ( connect_cancel == 1 ) {
                AbortIO( (struct IORequest *)SerialIO );
                WaitIO( (struct IORequest *)SerialIO );
                return 1;
            }

            count++;
        }

        if ( res != 0 ) {
            sprintf( error_message, "Failed to sync (1): %d", res );
            return -EINVAL;
        } else {
            if ( buf[0] == 'A' ) {
#ifndef GUI
                if ( BE_VERBOSE ) {
                    printf( "Client received answer\n" );
                }
#endif
                break;
            } else {
                invalid_count++;

                if ( invalid_count > 100 ) {
                    sprintf( error_message, "Got too many invalid bytes, maybe wrong speed settings?" );
                    return -EINVAL;
                }
            }
        }
    }

    while ( TRUE ) {
        SerialIO->IOSer.io_Length   = 1;
        SerialIO->IOSer.io_Data     = (APTR)"B";
        SerialIO->IOSer.io_Command  = CMD_WRITE;
        res = DoIO((struct IORequest *)SerialIO);

        if ( res == SerErr_NoDSR ) {
            continue;
        } else if ( res != 0 ) {
            sprintf( error_message, "Failed to sync (2): %d", res );
            return -EINVAL;
        } else {
            break;
        }
    }

#ifndef GUI
    if ( BE_VERBOSE ) {
        printf( "Client sync answer\n" );
    }
#endif

    while ( TRUE ) {
        SerialIO->IOSer.io_Length   = 1;
        SerialIO->IOSer.io_Data     = (APTR)&buf[0];
        SerialIO->IOSer.io_Command  = CMD_READ;
        res = DoIO((struct IORequest *)SerialIO);

        if ( res != 0 ) {
            sprintf( error_message, "Failed to sync (3): %d", res );
            return -EINVAL;
        } else {
            if ( buf[0] == 'C' ) {
#ifndef GUI
                if ( BE_VERBOSE ) {
                    printf( "Client received second answer" );
                }
#endif
                break;
            }
        }
    }

    while ( TRUE ) {
        SerialIO->IOSer.io_Length   = 1;
        SerialIO->IOSer.io_Data     = (APTR)"D";
        SerialIO->IOSer.io_Command  = CMD_WRITE;
        res = DoIO((struct IORequest *)SerialIO);

        if ( res == SerErr_NoDSR ) {
            continue;
        } else if ( res != 0 ) {
            sprintf( error_message, "Failed to sync (4): %d", res );
            return -EINVAL;
        } else {
            break;
        }
    }

#ifndef GUI
    if ( BE_VERBOSE ) {
        printf( "Client synced\n" );
    }
#endif

    return 0;
}

static int ser_transfer_block( void )
{
    LONG written = 0;
    LONG data_read = 0;
    LONG input_bytes = 0;
    const LONG output_bytes = outblock_used + outblock_base;

    putuint16( outblock, output_bytes );

    // start transfer by sending a single byte so the PC side does not
    // start sending data until we are ready within this receive
    // loop. Otherwise system activity might prevent incoming data
    // from being buffered (SerErr_LineErr)

    while ( TRUE ) {
        LONG res;
        SerialIO->IOSer.io_Length   = 1;
        SerialIO->IOSer.io_Data     = (APTR)"T";
        SerialIO->IOSer.io_Command  = CMD_WRITE;
        res = DoIO((struct IORequest *)SerialIO);

        if ( res == SerErr_NoDSR ) {
            //printf( "Not ready, try to repeat (%d)\n", SerialIO->IOSer.io_Actual );
        } else if ( res != 0 ) {
            sprintf( error_message, "Failed to write data" );
            return -EINVAL;
        } else {
            break;
        }
    }

    while ( TRUE ) {
        LONG oblock = MIN( APCCOMM_SERIAL_BLOCKSIZE, output_bytes - written );
        LONG iblock = MIN( APCCOMM_SERIAL_BLOCKSIZE, input_bytes - data_read );
        LONG res = 0;

        // for first round only read and write the u16 length field
        if ( data_read == 0 ) {
            oblock = 2;
            iblock = 2;
        }

        if ( oblock == 0 && iblock == 0 ) {
            break;
        }

        while ( oblock > 0 ) {
            SerialIO->IOSer.io_Length   = oblock;
            SerialIO->IOSer.io_Data     = (APTR)(outblock + written);
            SerialIO->IOSer.io_Command  = CMD_WRITE;
            res = DoIO((struct IORequest *)SerialIO);

            if ( res == SerErr_NoDSR ) {
                //printf( "Not ready, try to repeat (%d)\n", SerialIO->IOSer.io_Actual );
                oblock -= SerialIO->IOSer.io_Actual;
                written += SerialIO->IOSer.io_Actual;
            } else if ( res != 0 ) {
                sprintf( error_message, "Failed to write data" );
                return -EINVAL;
            } else {
                written += SerialIO->IOSer.io_Actual;
                oblock -= SerialIO->IOSer.io_Actual;
            }
        }

        /* SerialIO->IOSer.io_Command  = SDCMD_QUERY; */
        /* DoIO( (struct IORequest *)SerialIO ); */
 
        /* UWORD Serial_Status = SerialIO->io_Status; */
        /* ULONG Unread_Chars = SerialIO->IOSer.io_Actual; */

        while ( iblock > 0 ) {
            SerialIO->IOSer.io_Length   = iblock;
            SerialIO->IOSer.io_Data     = (APTR)(inblock + data_read);
            SerialIO->IOSer.io_Command  = CMD_READ;
            res = DoIO((struct IORequest *)SerialIO);

            if ( res != 0 ) {
                // e.g. SerErr_BufOverflow
                if ( res == SerErr_LineErr ) {
                    sprintf( error_message, "Failed to read data: %d (hardware overrun)", res );
                } else {
                    sprintf( error_message, "Failed to read data: %d", res );
                }
                return -EINVAL;
            }

            data_read += SerialIO->IOSer.io_Actual;
            iblock -= SerialIO->IOSer.io_Actual;
        }

        if ( input_bytes == 0 ) {
            unsigned short v = getuint16( inblock );

            input_bytes = v;

            if ( input_bytes < 2 || input_bytes > inblock_size ) {
                sprintf( error_message, "Error: invalid announced input bytes" );
                return -EINVAL;
            }
        }
    }

    inblock_actual_size = input_bytes;

    return 0;
}

static int ser_close( void )
{
    if ( DevOpened ) {
        if ( ! CheckIO( (struct IORequest *)SerialIO ) )  {
            AbortIO( (struct IORequest *)SerialIO );
        }
        WaitIO( (struct IORequest *)SerialIO );
        CloseDevice( (struct IORequest *)SerialIO );
    }

    if ( SerialIO ) {
        if ( osversion < 36 ) {
            DeleteExtIO( (struct IORequest*)SerialIO );
        } else {
            DeleteIORequest( SerialIO );
        }
        SerialIO = NULL;
    }

    if ( SerialMP ) {
        if ( osversion < 36 ) {
            DeletePort( SerialMP );
        } else {
            DeleteMsgPort( SerialMP );
        }
        SerialMP = NULL;
    }
    return 0;
}


int am_init_port( const struct apccomm_transfer_config *config )
{
    if ( ! config ) {
        return -EINVAL;
    }

    if ( config->mode == APCCOMM_PARALLEL ) {
        if ( initParPort() != 0 ) {
            return -EINVAL;
        }

        inblock_size = TRANSFERBLOCKSIZE/2;
        outblock_size = TRANSFERBLOCKSIZE;
        inblock_base = 0;
        outblock_base = 0;

        am_sync = synchro;
        am_transferblock = transferblock;
        am_cleanup = am_cleanup_par;
    } else if ( config->mode == APCCOMM_SERIAL ) {
        int res = ser_init( config );

        if ( res != 0 ) {
            return -EINVAL;
        }

        inblock_size = TRANSFERBLOCKSIZE;
        outblock_size = TRANSFERBLOCKSIZE;
        inblock_base = 2;
        outblock_base = 2;

        am_sync = ser_sync;
        am_transferblock = ser_transfer_block;
        am_cleanup = ser_close;
    } else {
        sprintf( error_message, "Invalid transfer mode %d", config->mode );
        return -EINVAL;
    }

    return 0;
}
