/* apccomm_pc_main.c
   Part of "APCComm" 
   Copyright (C) 2000-2023 Ralf Hoffmann
   Contact: ralf@boomerangsworld.de

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
  
*/

#define _GNU_SOURCE

#include "apccomm_pc.h"
#include "apccomm_pc_tr.h"
#include <getopt.h>
#include <ctype.h>

struct apccomm_transfer_config config;
static char *config_dir = NULL;

static char *get_config_dir( void )
{
    char *config_dir = NULL;
    char *slash;
    
    char *xdg_home = getenv( "XDG_CONFIG_HOME" );
    if ( xdg_home && strlen( xdg_home ) > 0 ) {
        size_t l  = strlen( xdg_home ) + 1 + strlen( "apccomm" ) + 1;
        config_dir = malloc( l );
        if ( ! config_dir ) return NULL;
        snprintf( config_dir, l, "%s/%s", xdg_home, "apccomm" );
    } else {
        const char *homestr = getenv( "HOME" );
        if ( ! homestr ) return NULL;

        size_t l1 = strlen( homestr ) + 1 + strlen( ".config" ) + 1;
        size_t l2 = l1 + 1 + strlen( "apccomm" ) + 1;

        config_dir = malloc( l2 );
        if ( ! config_dir ) return NULL;

        snprintf( config_dir, l2, "%s/.config", homestr );

        if ( access( config_dir, R_OK | W_OK | X_OK ) == 0 ) {
            snprintf( config_dir, l2, "%s/.config/apccomm", homestr );
        } else {
            snprintf( config_dir, l2, "%s/.apccomm", homestr );
        }
    }

    if ( ! config_dir ) return NULL;

    for ( slash = strchr( config_dir, '/' );
          slash != NULL;
          slash = strchr( slash + 1, '/' ) ) {

        *slash = '\0';

        if ( strlen( config_dir ) > 0 ) {
            if ( access( config_dir, F_OK ) != 0 ) {
                if ( mkdir( config_dir, 0777 ) != 0 ) {
                    fprintf( stderr, "Failed to create config dir component %s\n", config_dir );
                    free( config_dir );
                    return NULL;
                }
            }
        }

        *slash = '/';
    }

    if ( access( config_dir, F_OK ) != 0 ) {
        if ( mkdir( config_dir, 0777 ) != 0 ) {
            fprintf( stderr, "Failed to create config dir %s\n", config_dir );
            free( config_dir );
            return NULL;
        }
    }

    return config_dir;
}

static int save_settings( const struct apccomm_transfer_config *config,
                          int ignore_prot,
                          int verbose )
{
    int retval = 0;

    if ( ! config_dir ) return -ENOENT;

    size_t l = strlen( config_dir ) + 1 + strlen( "apccomm.conf" ) + 1;
    char *cfgfile = malloc( l );
    if ( ! cfgfile ) return -ENOMEM;

    snprintf( cfgfile, l, "%s/apccomm.conf", config_dir );

    if ( BE_VERBOSE ) {
        printf( "Saving settings to %s\n", cfgfile );
    }

    FILE *o_fh = fopen( cfgfile, "w" );
    if ( o_fh ) {
        char buf[512];

        if ( config->mode == APCCOMM_PARALLEL ) {
            if ( strlen( config->pc_usedevice ) > 0 ) {
                snprintf( buf, sizeof( buf ), "PORT=PARALLEL\nDEVICE=%s\n",
                          config->pc_usedevice );
            } else if ( config->pc_useport != 0 ) {
                snprintf( buf, sizeof( buf ), "PORT=PARALLEL\nPORTNUM=%d\n",
                          config->pc_useport );
            } else {
                snprintf( buf, sizeof( buf ), "PORT=PARALLEL\n" );
            }
        } else {
            snprintf( buf, sizeof( buf ), "PORT=SERIAL\nDEVICE=%s\nSPEED=%d\n",
                      config->pc_usedevice,
                      config->serial_speed );
        }

        if ( fwrite( buf, strlen( buf ), 1, o_fh ) != 1 ) {
            fprintf( stderr, "Failed to write settings to %s\n", cfgfile );
            unlink( cfgfile );
        }
        fclose( o_fh );
    } else {
        fprintf( stderr, "Failed to open settings file %s\n", cfgfile );
        retval = -EINVAL;
    }
    
    free( cfgfile );

    return retval;
}

static int load_settings( struct apccomm_transfer_config *config,
                          int *ignore_prot,
                          int *verbose )
{
    if ( ! config_dir ) return -ENOENT;

    size_t l = strlen( config_dir ) + 1 + strlen( "apccomm.conf" ) + 1;
    char *cfgfile = malloc( l );
    if ( ! cfgfile ) return -ENOMEM;

    snprintf( cfgfile, l, "%s/apccomm.conf", config_dir );

    FILE *i_fh = fopen( cfgfile, "r" );
    if ( i_fh ) {
        ssize_t nread;
        char *line = NULL;
        size_t len = 0;

        while ( ( nread = getline( &line, &len, i_fh ) ) != -1 ) {
            if ( strncmp( line, "PORT=", 5 ) == 0 ) {
                if ( strncmp( line + 5, "PARALLEL", 8 ) == 0 ) {
                    config->mode = APCCOMM_PARALLEL;
                } else if ( strncmp( line + 5, "SERIAL", 6 ) == 0 ) {
                    config->mode = APCCOMM_SERIAL;
                }
            } else if ( strncmp( line, "SPEED=", 6 ) == 0 ) {
                int s = atoi( line + 6 );
                switch ( s ) {
                    case 9600:
                    case 19200:
                    case 38400:
                    case 57600:
                    case 115200:
                        config->serial_speed = s;
                        break;
                    default:
                        break;
                }
            } else if ( strncmp( line, "DEVICE=", 7 ) == 0 ) {
                char *newline = strchr( line + 7, '\n' );
                if ( newline ) {
                    *newline = '\0';
                }
                snprintf( config->pc_usedevice,
                          sizeof( config->pc_usedevice ),
                          "%s",
                          line + 7 );
            } else if ( strncmp( line, "PORTNUM=", 8 ) == 0 ) {
                char *endptr;
                long res = strtol( line + 8, &endptr, 0 );

                if ( *endptr == '\0' ||
                     *endptr == '\n' ) {
                    config->pc_useport = (int)res;
                }
            }

        }

        free( line );

        fclose( i_fh );
    }

    return 0;
}

int main(int argc, char *argv[])
{
  int s1;
  int o1,o2,o3, c;
  short *realargs;
  int sendargs = 0;      /* Wenn es mal Argumente gibt, die nicht das
                            das Ende des Programmes verursachen... */
  unsigned long int ul_arg;
  char *endptr;
  int store_settings = 0;

  memset( &config, 0, sizeof( config ) );
  config.mode = APCCOMM_PARALLEL;

  config_dir = get_config_dir();

  load_settings( &config,
                 &ignore_prot,
                 &verbose );

  while ( 1 ) {
    int option_index;
    static struct option long_options[] = {
      { "help", no_argument, NULL, 'h' },
      { "version", no_argument, NULL, 'v' },
      { "verbose", no_argument, NULL, 'V' },
      { "quiet", no_argument, NULL, 'q' },
      { "ignore_protection", no_argument, NULL, 'i' },
      { "port", required_argument, NULL, 'p' },
      { "serial", required_argument, NULL, 's' },
      { "license", no_argument, NULL, 0 },
      { "speed", required_argument, NULL, 257 },
      { "save-settings", no_argument, NULL, 258 },
      { NULL, 0, NULL, 0 }
    };

    c = getopt_long( argc, argv, "hvVqip:s:", long_options, &option_index );
    if ( c == -1 ) break;

    switch( c ) {
    case 0:
      if ( strcmp( long_options[option_index].name, "license" ) == 0 ) {
	printf( aboutStr, MAJOR, MINOR, PATCH );
	printf( "\n" );
	exit( 0 );
      }
      break;
    case 'p':
        config.mode = APCCOMM_PARALLEL;
        if ( strcmp( optarg, "lpt1" ) == 0 ) {
            config.pc_useport = LPT1;
            config.pc_usedevice[0] = '\0';
        } else if ( strcmp( optarg, "lpt2" ) == 0 ) {
            config.pc_useport = LPT2;
            config.pc_usedevice[0] = '\0';
        } else if ( strncmp( optarg, "0x", 2 ) == 0 ) {
            ul_arg = strtoul( &optarg[2], &endptr, 16 );
            if (*endptr || !ul_arg  ||  ul_arg > 0xffff) {
                fprintf( stderr, "Invalid port base address!\n"
                         "Allowed range : 0x0001 - 0xFFFF (eg. 0x378, 0x278, 0x3BC, 0xA200, ...)\n" );
                exit( 1 );
            }
            config.pc_useport = (int) ul_arg;
            config.pc_usedevice[0] = '\0';
        } else if ( strncmp( optarg, "/dev/", 5 ) == 0 ) {
            snprintf( config.pc_usedevice,
                      sizeof( config.pc_usedevice ),
                      "%s",
                      optarg );
        } else {
            fprintf( stderr, "Invalid port! The possibilities are: lpt1, lpt2,\n"
                     "the port base address in HEX (e.g. 0x378, 0x278, 0x3BC, 0xA200, ...)\n"
                     "or the parport device (eg. /dev/parport0)\n" );
            exit( 1 );
        }
        break;
    case 's':
        config.mode = APCCOMM_SERIAL;
        snprintf( config.pc_usedevice,
                  sizeof( config.pc_usedevice ),
                  "%s",
                  optarg );
        if ( config.serial_speed == 0 ) {
            config.serial_speed = 57600;
        }
        break;
    case 'h':
      printf( "\nAPCComm by Ralf Hoffmann\n" );
      printf( "  Usage: %s [Option]... [<filename>]...\n  Transfer given files to the amiga\n\n", argv[0] );
      printf( "   -h, --help\t\t\tShow this help\n" );
      printf( "   -v, --version\t\tShow program version\n" );
      printf( "   -p, --port=<port>\t\tuse parallel port <port>\n" );
      printf( "   \t\t\t\t<port> can be lpt1, lpt2, the base address\n" );
      printf( "   \t\t\t\tin HEX (e.g. 0x378, 0x278, 0x3BC, 0xA200, ...)\n" );
      printf( "   \t\t\t\tor the parport device file (e.g. /dev/parport0)\n" );
      printf( "   -s, --serial=<port>\t\tuse given serial port\n" );
      printf( "   --speed=<mode>\t\tselect serial speed (slow,normal,fast or a\n\t\t\t\tnumber)\n" );
      printf( "   --save-settings\t\tsave settings\n" );
      printf( "   -V, --verbose\t\tverbose output\n" );
      printf( "   -q, --quiet\t\t\tno output (except errors)\n" );
      printf( "   -i, --ignore_protection\tignore file protection for receiving\n" );
      printf( "   --license\t\t\tShow program license\n" );
      printf( "  When starting without args, APCComm will receive files from the Amiga\n" );
      exit( 0 );
      break;
    case 'v':
      printf( "\nAPCComm by Ralf Hoffmann\n  Version %d.%d.%d\n", MAJOR, MINOR, PATCH );
      printf( "\n  Program for transfering files between Amiga and PC\n" );
      printf( "\n  This is free software with ABSOLUTELY NO WARRANTY\n" );
      printf( "  For details use --license\n\n" );
      printf( "%s", aboutStr2 );
      printf( "\n" );
      exit( 0 );
      break;
    case 'V':
      verbose = 1;
      break;
    case 'q':
      verbose = -1;
      break;
    case 'i':
      ignore_prot = 1;
      break;
    case '?':
      break;
    case ':':
      break;
        case 257:
            if ( strcmp( optarg, "slow" ) == 0 ) {
                config.serial_speed = 9600;
            } else if ( strcmp( optarg, "fast" ) == 0 ) {
                config.serial_speed = 115200;
            } else {
                config.serial_speed = 57600;

                if ( strlen( optarg ) > 0 &&
                     isdigit( optarg[0] ) ) {
                    int v = atoi( optarg );

                    switch ( v ) {
                        case 9600:
                        case 19200:
                        case 38400:
                        case 57600:
                        case 115200:
                            config.serial_speed = v;
                            break;
                        default:
                            fprintf( stderr, "Unsupported speed %s\n", optarg );
                            exit( 1 );
                    }
                }
            }
            break;
        case 258:
            store_settings = 1;
            break;
    default:
      break;
    }
  }

  if ( store_settings ) {
      if ( save_settings( &config, ignore_prot, verbose ) != 0 ) {
          fprintf( stderr, "failed to store settings\n" );
          exit( 1 );
      }
  }

  realargs = (short*)malloc( sizeof( short ) * argc );
  for(s1=0;s1<argc;s1++) {
    realargs[s1] = 0;
  }

  while ( optind < argc ) {
    sendargs++;
    realargs[optind++] = 1;
  }

  if(init_transferblocks()!=0) {
    fprintf(stderr,"Not enough mem in main()!\nExiting\n");
    free( realargs );
    exit(1);
  }

  if ( pc_init_port( &config ) != 0 ) {
    free( realargs );
    exit(0);
  }

  if ( ! BE_QUIET ) printf("Waiting for Amiga:");
  fflush(stdout);
  pc_sync();
  if ( ! BE_QUIET ) printf("\nConnection established\n");

  /* Prepare the outblock */
  outputInt( 0, PROTOCOL_MAJOR );
  outputInt( 4, PROTOCOL_MINOR );
  outputInt( 8, PROTOCOL_PATCH );
  if(sendargs>0) outputInt( 12, 1 );
  else outputInt( 12, 0 );
  setOutputSize( 16 );

  /* Inform the other side */
  int res = pc_transferblock();
  if ( res != 0 ) {
      printf( "Error during initial data transfer\n" );
      free( realargs );
      exit(1);
  }

  /* Now check the other version */
  o1 = inputInt( 0 );
  o2 = inputInt( 4 );
  o3 = inputInt( 8 );
  if( (o1!=PROTOCOL_MAJOR) || (o2!=PROTOCOL_MINOR) ) {
    printf("Wrong version at the other side!\n");
    printf("  Expected %d.%d.x ,  got %d.%d.%d\n",PROTOCOL_MAJOR,PROTOCOL_MINOR,o1,o2,o3);
    free( realargs );
    exit(1);
  }

  /* Version ok, sending conflict ? */
  o1 = inputInt( 12 );
  if( (o1==1) && (sendargs>0) ) {
    printf("Both sides want to send!\n");
    printf("  This is not supported!\n");
    free( realargs );
    exit(1);
  }

  if( (o1==0) && (sendargs==0) ) {
    printf("Nothing to do!\n");
    free( realargs );
    exit(1);
  }

  if(sendargs>0) {
    send(argc,argv,realargs);
  } else {
    receive();
  }

  pc_cleanup();

  free(inblock);
  free(outblock);
  free( realargs );
  return 0;
}
