/* apccomm_pc_tr.c
   Part of "APCComm" 
   Copyright (C) 2000-2023 Ralf Hoffmann
   Contact: ralf@boomerangsworld.de

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
  
*/

#include "apccomm_pc_tr.h"
#include "apccomm_pc.h"

int recvcommand(apc_command_t *com,char **arg,int *mode)
{
  apc_command_t tc;
  char *narg;

  setOutputSize( 0 );
  if ( pc_transferblock() != 0 ) {
      fprintf( stderr, "Transmission error!\n" );
      return 1;
  }
  tc = (apc_command_t)inputInt( 0 );
  switch(tc) {
    case APC_COMMAND_ERROR:
      *com=tc;
      if(arg!=NULL) *arg=NULL;
      if(mode!=NULL) *mode=0;
      break;
    case APC_COMMAND_ENTERDIR:
        if(arg!=NULL) {
            int l;

            *mode = inputInt( 4 );
            l = inputInt( 8 );
            int rem = inputSize() - 12;
            if ( l > rem ) l = rem;
            narg=(char*)malloc(l+1);
            if(narg!=NULL) {
                *com=tc;
                const unsigned char *data = inputData( 12 );
                strncpy( narg, (char*)( data ), l );
                narg[l]='\0';
                *arg=narg;
            } else return 1;
        } else return 1;
        break;
    case APC_COMMAND_LEAVEDIR:
      *com=tc;
      if(arg!=NULL) *arg=NULL;
      if(mode!=NULL) *mode=0;
      break;
    case APC_COMMAND_FILETRANSFER:
      *com=tc;
      if(arg!=NULL) *arg=NULL;
      if(mode!=NULL) *mode=0;
      return recvfile();
      break;
    case APC_COMMAND_FINISHED:
      *com=tc;
      if(arg!=NULL) *arg=NULL;
      if(mode!=NULL) *mode=0;
      break;
    case APC_COMMAND_NOP:
      *com=tc;
      if(arg!=NULL) *arg=NULL;
      if(mode!=NULL) *mode=0;
      break;
    case APC_COMMAND_ABORT:
      *com=tc;
      if(arg!=NULL) *arg=NULL;
      if(mode!=NULL) *mode=0;
      break;
    case APC_COMMAND_SKIPFILE:
      *com=tc;
      if(arg!=NULL) *arg=NULL;
      if(mode!=NULL) *mode=0;
      break;
    default:
      fprintf( stderr, "Unknown command %d received!\n", tc );
      return 1;
      break;
  }
  return 0;
}

int recvfile()
{
  struct timeval tvs,tvi;
  int secs,usecs;
  int outchecksum,inchecksum,checksum,wrong,s1,tc;
  double d1;
  int getfile;
  int size=0,recbytes,count,l;
  char outbuf[128],recvbuf[128],*cwd,*tstr=NULL;
  int cwdsize;
  char *inarg=NULL;
  int ret;
  int mode,tm;
  int update_rate = 19; // default for parallel

  if ( config.mode == APCCOMM_SERIAL ) {
      update_rate = 5;
  }

  size = inputInt( 4 );
  mode = inputInt( 8 );
  l = inputInt( 12 );

  int rem = inputSize() - 16;
  const unsigned char *data;

  inarg=(char*)malloc(l+1);
  if(inarg==NULL) return 1;

  data = inputData( 16 );
  strncpy( inarg, (char*)( data ), min( l, rem ) );
  inarg[min( l, rem )]='\0';

  
  if ( ! BE_QUIET ) printf("receiving file \"%s\", size %d\n",inarg,size);
  ret = init_incoming_file( inarg, size );

  getfile=1;

  checksum=0;

  sprintf(outbuf,"%d",size);
  sprintf( recvbuf, "RECV: %%%dd/%%%dd (%%6.2f%%%%)",
           (int)strlen(outbuf),
           (int)strlen(outbuf) );
    
  inchecksum=outchecksum=0;
  recbytes=0;
  wrong=0;
  tc = 0;

  if((ret>0)||(ret==-2)) {
    fprintf( stderr, "Can't create %s in %s, aborting\n", inarg, basedir );
    sendcommand(APC_COMMAND_ABORT,NULL, 0 );
  } else if(ret==-1) {
    sendcommand(APC_COMMAND_SKIPFILE,NULL, 0 );
    ret = 0;
  } else if(ret==0) {
    sendcommand(APC_COMMAND_NOP,NULL, 0 );

    gettimeofday(&tvs,NULL);
    for(count=0;getfile==1;count++) {
      if(recbytes<size) {
        setOutputSize( 4 );
        if ( cancel_transfer ) {
            outputTrailingInt( 0xffffffee );
            wrong = 4;
        } else if(wrong==0) outputTrailingInt( 0 );
        else outputTrailingInt( 0xffffffff );

        if(getfile==1) {
            if ( pc_transferblock() != 0 ) wrong = 5;
            tc = inputTrailingInt();
            if(tc==0xffffffff) wrong=2;
            else if (tc == 0xffffffee) wrong = 4;
        }
        if(wrong!=0) break;

        rem = inputSize();

        s1=size-recbytes;
        if(s1>rem - 4) s1=rem - 4;
        else getfile=2;

        writefrombuffer( inputData( 0 ), s1, &inchecksum );

        recbytes+=s1;
        checksum=inchecksum-tc;
        if(checksum!=0) wrong=1;
      } else getfile = 2;

      if ( ( count == update_rate ) || ( getfile != 1 ) ) {
          count=0;
          if ( size )
              sprintf(outbuf,recvbuf,recbytes,size,((double)recbytes/size)*100);
          else
              sprintf(outbuf,recvbuf,0,0,100.0);
          if ( ! BE_QUIET ) printf("\r%s",outbuf);
          fflush(stdout);
      }
    }
    gettimeofday(&tvi,NULL);
    if ( BE_VERBOSE ) printf("\nTransfer finished");
    else if ( ! BE_QUIET ) printf("\n");
    if(wrong==1) {
      if ( BE_VERBOSE ) printf("\n");
      fprintf( stderr, "Error while transfer\nI have received wrong data!\n" );
      fprintf( stderr, "Please repeat transmission!\n" );
    } else if(wrong==2) {
      if ( BE_VERBOSE ) printf("\n");
      fprintf( stderr, "Error while transfer\nThe Amiga reports data loss!\n" );
      fprintf( stderr, "Please repeat transmission!\n" );
    } else if(wrong==5) {
      if ( BE_VERBOSE ) printf("\n");
      fprintf( stderr, "Error while transmitting data\n" );
    } else if ( wrong == 4 ) {
      if ( BE_VERBOSE ) printf("\n");
      ret = -1;
    } else {
      if(getfile>0) {
	secs=tvi.tv_sec-tvs.tv_sec;
	usecs=tvi.tv_usec-tvs.tv_usec;
	if(usecs<0) {
	  secs--;
	  usecs+=1000000;
	}
	d1=size;
	d1*=1000000;
	d1/=(secs*1000000.0+usecs);
	d1/=1024;
	if ( BE_VERBOSE ) printf(", file received with %.2g KBytes/s\n",d1);

	tfl.files++;
	tfl.bytes += size;
	tfl.tvs.tv_sec += secs;
	tfl.tvs.tv_usec += usecs;
	if ( tfl.tvs.tv_usec >= 1000000 ) {
	  tfl.tvs.tv_sec++;
	  tfl.tvs.tv_usec -= 1000000;
	}
      }
    }
    close_incoming_file();
    
    /* File is stored in the current dir when basedir is NULL
     * File is stored relative to current dir in basedir when basedir starts NOT with "/"
     * File is stored in basedir when basedir starts with "/"
     */

    /* First get the cwd */
    cwdsize=1024;
    cwd=(char*)malloc(cwdsize);
    if(cwd!=NULL) {
      while(getcwd(cwd,cwdsize)==NULL) {
	if(errno==ERANGE) {
	  free(cwd);
	  cwdsize*=2;
	  cwd=(char*)malloc(cwdsize);
	  if(cwd==NULL) break;
	} else {
	  free(cwd);
	  cwd=NULL;
	}
      }
    }
    
    /* now check all 3 cases */
    
    if ( basedir == NULL ) {
      
      /* just in the cwd */
      
      if(cwd!=NULL) {
	if(strlen(cwd)<2) {
	  tstr=(char*)malloc(strlen(cwd)+strlen(infile.name)+1);
	  if(tstr!=NULL) sprintf(tstr,"%s%s",cwd,infile.name);
	} else {
	  tstr=(char*)malloc(strlen(cwd)+1+strlen(infile.name)+1);
	  if(tstr!=NULL) sprintf(tstr,"%s/%s",cwd,infile.name);
	}
      }
    } else {
      
      /* at least relative to basedir */
      
      if ( basedir[0] != '/' ) {
	
	/* basedir is relative */
	
	if ( cwd[strlen( cwd ) - 1] == '/' ) {
	  tstr = (char*) malloc( strlen( cwd ) + strlen( basedir ) + strlen( infile.name ) + 1 );
	  if ( tstr != NULL ) sprintf( tstr, "%s%s%s", cwd, basedir, infile.name );
	} else {
	  tstr = (char*) malloc( strlen( cwd ) + 1 + strlen( basedir ) + strlen( infile.name ) + 1 );
	  if(tstr!=NULL) sprintf( tstr, "%s/%s%s", cwd, basedir, infile.name );
	}
      } else {
	tstr = (char*) malloc( strlen( basedir ) + strlen( infile.name ) + 1 );
	if ( tstr != NULL ) sprintf( tstr, "%s%s", basedir, infile.name );
      }
    }

    
    if(tstr!=NULL) {
      if ( BE_VERBOSE ) printf("Incoming file is stored at: %s\n",tstr);
      if ( wrong ) fprintf( stderr, "Wrong file is at %s\n", tstr );

      /* Apply mode
       * mode is the unix number but to be sure:
       */
      tm = 0;
      if ( mode & 0400 ) tm |= S_IRUSR;
      if ( mode & 0200 ) tm |= S_IWUSR;
      if ( mode & 0100 ) tm |= S_IXUSR;
      if ( mode & 0040 ) tm |= S_IRGRP;
      if ( mode & 0020 ) tm |= S_IWGRP;
      if ( mode & 0010 ) tm |= S_IXGRP;
      if ( mode & 0004 ) tm |= S_IROTH;
      if ( mode & 0002 ) tm |= S_IWOTH;
      if ( mode & 0001 ) tm |= S_IXOTH;
      
      if ( ! ignore_prot )
	if ( chmod( tstr, tm ) != 0 ) {
	  fprintf( stderr, "Can't restore file protection to %s\n", tstr );
      }

      free(tstr);
    }
    
    if ( cwd != NULL) free(cwd);
    free(infile.name);
  }
  free(inarg);
  return ret;
}

int sendfile(const char *name, int mode)
{
  struct timeval tvs,tvo;
  int secs,usecs;
  int outchecksum,wrong,tc;
  double d1;
  int sendfile;
  int sendbytes,ts,count;
  char outbuf[128],sendbuf[128];
  const char *outbasename=NULL;

  apc_command_t com;
  int ret=0;
  int update_rate = 19; // default for parallel

  if ( config.mode == APCCOMM_SERIAL ) {
      update_rate = 5;
  }

  if(init_outgoing_file(name)!=0) {
    fprintf( stderr, "can't open file %s!\n", name );
    return 1;
  } else {
    sendfile=1;
    if ( ! BE_QUIET ) printf("sending file \"%s\", size %d\n",name,outfile.size);
  }
  outputInt( 0, APC_COMMAND_FILETRANSFER );
  outputInt( 4, outfile.size );
  outputInt( 8, mode );

  outbasename=strrchr(name,'/');
  if(outbasename==NULL) outbasename=name;
  else {
    outbasename++;
  }

  int rem = maxOutputSize() - 16;
  int l = min( strlen( outbasename ),
               rem );
  outputInt( 12, l );
  strncpy( (char*)( outputData( 16 ) ), outbasename, l );
  setOutputSize( l + 16 );

  pc_transferblock();

  com = (apc_command_t)inputInt( 0 );
  if ( com != APC_COMMAND_ABORT ) {

    /* preparing progress display */
    sprintf(outbuf,"%d",outfile.size);
    sprintf( sendbuf, "SEND: %%%dd/%%%dd (%%6.2f%%%%)",
             (int)strlen(outbuf),
             (int)strlen(outbuf) );
    
    outchecksum=0;
    sendbytes=0;
    wrong=0;
    
    /* now synchronize with other side and check state */
    if(recvcommand(&com,NULL,NULL)==0) {
      if(com==APC_COMMAND_NOP) {
	outchecksum=0;
	sendbytes=0;
	wrong=0;
	gettimeofday(&tvs,NULL);
	for(count=0;sendfile==1;count++) {
        ts=readtobuffer(outputData( 0 ),maxOutputSize() - 4,&outchecksum);
        sendbytes+=ts;
        if ( ts < 1 ) sendfile=2;
        if ( ts < 0 ) wrong = 3;

        setOutputSize( ts + 4 );
        if ( cancel_transfer ) {
            outputTrailingInt( 0xffffffee );
            wrong = 4;
        } else if(wrong==0) outputTrailingInt( outchecksum );
        else outputTrailingInt( 0xffffffff );

        if ( ( sendfile == 1 ) || ( wrong != 0 ) ) {

            if ( pc_transferblock() != 0 ) wrong = 5;
            tc = inputTrailingInt();
            if(tc==0xffffffff) wrong=2;
            else if ( tc == 0xffffffee ) wrong = 4; /* abort */
        }
        if(wrong!=0) break;

        if ( ( count == update_rate ) || ( sendfile != 1 ) ) {
            count=0;
            if ( outfile.size )
                sprintf(outbuf,sendbuf,sendbytes,outfile.size,((double)sendbytes/outfile.size)*100);
            else
                sprintf(outbuf,sendbuf,0,0,100.0);
            if ( ! BE_QUIET ) printf("\r%s",outbuf);
            fflush(stdout);
        }
	}
	gettimeofday(&tvo,NULL);
	if ( BE_VERBOSE ) printf("\nTransfer finished");
	else if ( ! BE_QUIET ) printf("\n");
	if(wrong==1) {
	  if ( BE_VERBOSE ) printf("\n");
	  fprintf( stderr, "Error while transfer\nI have received wrong data!\n" );
	  fprintf( stderr, "Please repeat transmission!\n" );
	  fprintf( stderr, "Wrong file was %s\n", name );
	} else if(wrong==2) {
	  if ( BE_VERBOSE ) printf("\n");
	  fprintf( stderr, "Error while transfer\nThe Amiga reports data loss!\n" );
	  fprintf( stderr, "Please repeat transmission!\n" );
	  fprintf( stderr, "Wrong file was %s\n", name );
	} else if(wrong==3) {
	  if ( BE_VERBOSE ) printf("\n");
	  fprintf( stderr, "Error while reading file\n" );
	  fprintf( stderr, "Wrong file was %s\n", name );
	} else if(wrong==5) {
	  if ( BE_VERBOSE ) printf("\n");
	  fprintf( stderr, "Error while transmitting data\n" );
	  fprintf( stderr, "Wrong file was %s\n", name );
    } else if ( wrong == 4 ) {
        if ( BE_VERBOSE ) printf("\n");
        ret = -1;
	} else {
	  secs=tvo.tv_sec-tvs.tv_sec;
	  usecs=tvo.tv_usec-tvs.tv_usec;
	  if(usecs<0) {
	    secs--;
	    usecs+=1000000;
	  }
	  d1=outfile.size;
	  d1*=1000000;
	  d1/=(secs*1000000.0+usecs);
	  d1/=1024;
	  if ( BE_VERBOSE ) printf(", file sent with %.2g KBytes/s\n",d1);

	  tfl.files++;
	  tfl.bytes += outfile.size;
	  tfl.tvs.tv_sec += secs;
	  tfl.tvs.tv_usec += usecs;
	  if ( tfl.tvs.tv_usec >= 1000000 ) {
	    tfl.tvs.tv_sec++;
	    tfl.tvs.tv_usec -= 1000000;
	  }
	}
      } else {
	if(com==APC_COMMAND_SKIPFILE) {
	  if ( ! BE_QUIET ) printf("Skipping file!\n");
	} else if(com==APC_COMMAND_ABORT) {
	  ret=-1;
	} else {
	  fprintf( stderr, "Wrong command %d\n", com );
	  ret=1;
	}
      }
    } else {
      fprintf( stderr, "Error while receiving command!\n" );
      ret=1;
    }
  } else {
    ret = -1;
  }
  close_outgoing_file();
  free(outfile.name);
  return ret;
}

int sendcommand(apc_command_t com,const char *arg, int mode)
{
  if(com==APC_COMMAND_FILETRANSFER) {
    return sendfile(arg, mode);
  } else if(com==APC_COMMAND_ERROR) {
    outputInt( 0, com );
    setOutputSize( 4 );
    if ( pc_transferblock() != 0 ) {
        fprintf( stderr, "Transmission error!\n" );
        return 1;
    }
  } else if(com==APC_COMMAND_ENTERDIR) {
      int rem = maxOutputSize();
      if( strlen( arg ) + 4 + 4 + 4 <= rem ) {
          outputInt( 0, com );
          outputInt( 4, mode );

          int l = min( rem - 12,
                       strlen( arg ) );
          outputInt( 8, l );
          strncpy( (char*)( outputData( 12 ) ), arg, l );
          setOutputSize( l + 12 );
          if ( pc_transferblock() != 0 ) {
              fprintf( stderr, "Transmission error!\n" );
              return 1;
          }
    } else {
      fprintf( stderr, "Too long dirname!\n" );
      return 1;
    }
  } else if(com==APC_COMMAND_LEAVEDIR) {
    outputInt( 0, com );
    setOutputSize( 4 );
    if ( pc_transferblock() != 0 ) {
        fprintf( stderr, "Transmission error!\n" );
        return 1;
    }
  } else if(com==APC_COMMAND_FINISHED) {
    outputInt( 0, com );
    setOutputSize( 4 );
    if ( pc_transferblock() != 0 ) {
        fprintf( stderr, "Transmission error!\n" );
        return 1;
    }
  } else if(com==APC_COMMAND_NOP) {
    outputInt( 0, com );
    setOutputSize( 4 );
    if ( pc_transferblock() != 0 ) {
        fprintf( stderr, "Transmission error!\n" );
        return 1;
    }
  } else if(com==APC_COMMAND_ABORT) {
    outputInt( 0, com );
    setOutputSize( 4 );
    if ( pc_transferblock() != 0 ) {
        fprintf( stderr, "Transmission error!\n" );
        return 1;
    }
  } else if(com==APC_COMMAND_SKIPFILE) {
    outputInt( 0, com );
    setOutputSize( 4 );
    if ( pc_transferblock() != 0 ) {
        fprintf( stderr, "Transmission error!\n" );
        return 1;
    }
  } else {
    fprintf( stderr, "Unknown command %d to send!\n", com );
    return 1;
  }
  return 0;
}
