/* apccomm_pc_tr_lowlevel.c
   Part of "APCComm" 
   Copyright (C) 2022-2023 Ralf Hoffmann
   Contact: ralf@boomerangsworld.de

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
  
*/

#include "apccomm_pc_tr.h"
#include "apccomm_pc.h"
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <termios.h>
#include <inttypes.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/select.h>

#include <tavvva_ppdrv.h>

int (*pc_sync)( void ) = NULL;
int (*pc_transferblock)( void ) = NULL;
int (*pc_cleanup)( void ) = NULL;

int amiga_nibble;
int ack;

static struct termios original_options;
static short use_tpp = 0;
static int serial_fd = -1;

static int ser_close( void );

void enable_tpp_in_tr_core()
{
    use_tpp = 1;
}

inline void setack(int value)
{
  if ( use_tpp ) {
    tpp_write_pin(3, value);
  } else {
    change_pin(LP_PIN[3],(value==1)?LP_SET:LP_CLEAR);
  }
}

inline int getack()
{
  if ( use_tpp ) {
    tpp_read_pin(11,&ack);
  } else {
    ack=(pin_is_set(LP_PIN11)==0)?0:1;
  }
  return ack;
}

inline int getnibble()
{
  int i,i2;
  unsigned char status;
/*  i=(pin_is_set(LP_PIN10)==0)?0:8;
  i|=(pin_is_set(LP_PIN12)==0)?0:4;
  i|=(pin_is_set(LP_PIN13)==0)?0:2;
  i|=(pin_is_set(LP_PIN15)==0)?0:1;
  amiga_nibble=i;
  return i;*/
  if ( use_tpp ) {
    tpp_read_status (&status);
    i2  = 8 * stat_to_pin10(status) +
          4 * stat_to_pin12(status) +
          2 * stat_to_pin13(status) +
          1 * stat_to_pin15(status);
  } else {
    i=pin_is_set(LP_PIN10|LP_PIN12|LP_PIN13|LP_PIN15);
    if(i&LP_PIN10) i2=8; else i2=0;
    if(i&LP_PIN12) i2|=4;
    if(i&LP_PIN13) i2|=2;
    if(i&LP_PIN15) i2|=1;
  }

  amiga_nibble=i2;
  return i2;
}

inline void putvalue(int value)
{
  static unsigned char data;
  if ( use_tpp) {
    data = tpp_data_loopback();
    pin_to_data(4, value & 2, data);
    pin_to_data(6, value & 1, data);
    tpp_write_data(data);
  } else {
    if(value==3) set_pin(LP_PIN06|LP_PIN04);
    else if(value==0) clear_pin(LP_PIN06|LP_PIN04);
    else {
      change_pin(LP_PIN06,(value&1)?LP_SET:LP_CLEAR);
      change_pin(LP_PIN04,(value&2)?LP_SET:LP_CLEAR);
    }
  }
}

static int synchro()
{
  int count;
  setack(1);
  putvalue(0);
/*  while(getack()!=1);*/
  for(count=0;getack()!=1;count++) {
    usleep(1000);
    if(count%100==0) {
      if ( ! BE_QUIET ) printf(".");
      fflush(stdout);
    }
  }
  setack(0);
/*  while(getnibble()!=1);*/
  for(count=0;getnibble()!=1;count++) {
    usleep(1000);
    if(count%100==0) {
      if ( ! BE_QUIET ) printf(".");
      fflush(stdout);
    }
  }
  putvalue(1);
/*  while(getnibble()!=2);*/
  for(count=0;getnibble()!=2;count++) {
    usleep(1000);
    if(count%100==0) {
      if ( ! BE_QUIET ) printf(".");
      fflush(stdout);
    }
  }
  putvalue(2);
  setack(1);

  return 0;
}

int transferblock()
{
  int i,s1;
  int outbyte=0,inbyte,bytecyc,ti;

  bytecyc=3;
  for(i=0;i<inblock_size;i++) {
    s1=0;
    if(bytecyc==3) outbyte=outblock[i>>1];
    s1=outbyte>>((bytecyc--)*2);
    while(getack()!=0);
    putvalue(s1&0x3);
    inbyte=getnibble();
    setack(0);
    s1=outbyte>>((bytecyc--)*2);
    while(getack()!=1);
    putvalue(s1&0x3);
    ti=getnibble();
    ti<<=4;
    inbyte|=ti;
    setack(1);
    if(bytecyc<0) bytecyc=3;
    inblock[i]=(unsigned char)inbyte;
  }

  return 0;
}

static int pc_cleanup_par()
{
    if (tpp_active()) tpp_exit();

    return 0;
}

static int initParPort( const struct apccomm_transfer_config *config )
{
    int useport = LPT1;

    if ( strlen( config->pc_usedevice ) > 0 ) {
        if ( tpp_init( (char*)config->pc_usedevice ) ) {
            fprintf( stderr, "Failed to init parport\n" );
            return -EFAULT;
        }
    }

    if ( config->pc_useport != 0 ) {
        useport = config->pc_useport;
    }

    if (!tpp_active() && pin_init_user(useport) < 0) {
        return -1;
    }

    /* drop root privileges needed for access to parport */
    setuid(getuid());
    setgid(getgid());

    if (tpp_active()) {
        tpp_set_data_direction(0);
        enable_tpp_in_tr_core();
    } else {
        pin_output_mode(LP_DATA_PINS);
    }

    return 0;
}

static int ser_sync( void )
{
    int count = 0;

    if ( tcflush( serial_fd, TCIOFLUSH ) != 0 ) {
        fprintf( stderr, "Failed to flush queue: %s\n", strerror( errno ) );
        return -errno;
    }

    char buf[16];
    ssize_t res;
    const int initial_chars_per_second = 10;

    if ( BE_VERBOSE ) {
        printf( "Main start sync\n" );
    }

    while ( true ) {
        res = write( serial_fd, "A", 1 );
        if ( res < 0 && errno != EAGAIN ) {
            fprintf( stderr, "Failed to sync: %s\n", strerror( errno ) );
            return -errno;
        }

        tcdrain( serial_fd );

        res = read( serial_fd, buf, 1 );
        if ( res < 0 && errno != EAGAIN ) {
            fprintf( stderr, "Failed to sync: %s\n", strerror( errno ) );
            return -errno;
        } else if ( res < 0 && errno == EAGAIN ) {
        } else if ( res == 1 ) {
            if ( buf[0] == 'B' ) {
                if ( BE_VERBOSE ) {
                    printf( "Main received answer\n" );
                }
                break;
            }
        }

        usleep( 1000000 / initial_chars_per_second );
        if ( count % initial_chars_per_second == 0 ) {
            if ( ! BE_QUIET ) printf(".");
            fflush(stdout);
        }
        count++;
    }

    while ( true ) {
        res = write( serial_fd, "C", 1 );
        if ( res < 0 && errno != EAGAIN ) {
            fprintf( stderr, "Failed to sync: %s\n", strerror( errno ) );
            return -errno;
        } else if ( res == 1 ) {
            break;
        }
    }

    tcdrain( serial_fd );

    if ( BE_VERBOSE ) {
        printf( "Main sync answer\n" );
    }

    while ( true ) {
        res = read( serial_fd, buf, 1 );
        if ( res < 0 && errno != EAGAIN ) {
            fprintf( stderr, "Failed to sync: %s\n", strerror( errno ) );
            return -errno;
        } else if ( res < 0 && errno == EAGAIN ) {
        } else if ( res == 0 ) {
        } else {
            if ( buf[0] == 'D' ) {
                if ( BE_VERBOSE ) {
                    printf( "Main received second answer\n" );
                }
                break;
            }
        }
    }

    if ( BE_VERBOSE ) {
        printf( "Main synced\n" );
    }

    return 0;
}

#define MIN( a, b ) ((a) < (b) ? (a) : (b))

static int ser_transfer_block( void )
{
    ssize_t written = 0;
    ssize_t data_read = 0;
    fd_set rfds;
    ssize_t input_bytes = 0;
    const ssize_t output_bytes = outblock_used + outblock_base;

    putuint16( outblock, output_bytes );

    // wait for start transfer indication so we do not write data to
    // send until the Amiga is inside the receive loop
    while ( true ) {
        ssize_t res = 0;
        res = read( serial_fd,
                    inblock,
                    1 );

        if ( res < 0 && errno != EAGAIN ) {
            fprintf( stderr, "Failed to read: %s\n", strerror( errno ) );
            return -errno;
        } else if ( res < 0 && errno == EAGAIN ) {
            FD_ZERO( &rfds );
            FD_SET( serial_fd, &rfds );

            int sret = select( serial_fd + 1, &rfds, NULL, NULL, NULL );
            if ( sret == -1 && errno != EINTR ) {
                fprintf( stderr, "Failed to wait for data: %s\n", strerror( errno ) );
                return -errno;
            }
        } else if ( res == 1 ) {
            if ( inblock[0] != 'T' ) {
                fprintf( stderr, "Wrong start indicator: %d\n", inblock[0] );
                return -errno;
            }
            break;
        }
    }

    while ( true ) {
        ssize_t oblock = MIN( APCCOMM_SERIAL_BLOCKSIZE, output_bytes- written );
        ssize_t iblock = MIN( APCCOMM_SERIAL_BLOCKSIZE, input_bytes - data_read );
        ssize_t res = 0;

        // for first round only read and write the u16 length field
        if ( data_read == 0 ) {
            oblock = 2;
            iblock = 2;
        }

        if ( oblock == 0 && iblock == 0 ) {
            break;
        }

        // write first to fill OS buffers, read will block until the Amiga sends data
        while ( oblock > 0 ) {
            res = write( serial_fd,
                         outblock + written,
                         oblock );
            if ( res < 0 ) {
                fprintf( stderr, "Failed to write: %s\n", strerror( errno ) );
                return -errno;
            }
            //printf( "Wrote %lu data\n", res );

            oblock -= res;
            written += res;
        }

        tcdrain( serial_fd );

        while ( iblock > 0 ) {
            res = read( serial_fd,
                        inblock + data_read,
                        iblock );

            if ( res < 0 && errno != EAGAIN ) {
                fprintf( stderr, "Failed to read: %s\n", strerror( errno ) );
                return -errno;
            } else if ( res < 0 && errno == EAGAIN ) {
                FD_ZERO( &rfds );
                FD_SET( serial_fd, &rfds );

                int sret = select( serial_fd + 1, &rfds, NULL, NULL, NULL );
                if ( sret == -1 && errno != EINTR ) {
                    fprintf( stderr, "Failed to wait for data: %s\n", strerror( errno ) );
                    return -errno;
                }
            } else {
                //printf( "Got %lu(%lu) data\n", res, iblock );

                data_read += res;
                iblock -= res;
            }
        }

        //printf( "Remaining: %lu/%lu\n", 1024 - written, 1024 - data_read );

        if ( input_bytes == 0 ) {
            unsigned short v = getuint16( inblock );

            input_bytes = v;

            if ( input_bytes < 2 || input_bytes > inblock_size ) {
                fprintf( stderr, "Error: invalid announced input bytes\n" );
                return -errno;
            }
        }
    }

    inblock_actual_size = input_bytes;

    return 0;
}

static void restore_settings( void )
{
    if ( serial_fd != -1 ) {
        tcflush( serial_fd, TCIOFLUSH );

        if ( tcsetattr( serial_fd, TCSAFLUSH, &original_options ) != 0 ) {
            fprintf( stderr, "Failed to restore serial settings\n" );
        } else {
            if ( BE_VERBOSE ) {
                printf( "Restored settings\n" );
            }
        }
    }
}

static int ser_close()
{
    // unfortunately I dont know of any way of ensuring the data is
    // really sent over the line. For really low speeds like 9600, it
    // happen that the last few bytes (<10) are not seen on the other
    // side, despite sycn and drain. So I use a final sleep of 100ms.
    fsync( serial_fd );
    tcdrain( serial_fd );
    usleep( 100000 );
    restore_settings();
    return close( serial_fd );
}

static void sighandler( int signo )

{
    if ( BE_VERBOSE ) {
        printf( "Signal handler called\n" );
    }

    if ( ! cancel_transfer ) {
        printf( "\nCancelling transfer. Press Ctrl-c again to force quit\n" );
        cancel_transfer = 1;
    } else {
        if ( serial_fd != -1 ) {
            restore_settings();
        }
        _exit( EXIT_FAILURE );
    }
}

static int ser_init( const struct apccomm_transfer_config *config )
{
    const char *tty_name = "/dev/ttyUSB0";
    struct termios options;

    if ( strlen( config->pc_usedevice ) > 0 ) {
        tty_name = config->pc_usedevice;
    }

    serial_fd = open( tty_name, O_RDWR | O_NOCTTY | O_NDELAY );

    if ( serial_fd == -1 ) {
        fprintf( stderr, "Failed to open device %s: %s\n", tty_name, strerror( errno ) );
        return -errno;
    }

    if ( fcntl( serial_fd, F_SETFL, FNDELAY ) != 0 ) {
        fprintf( stderr, "Failed to set to non-blocking\n" );
        ser_close();

        return -errno;
    }

    if ( tcgetattr( serial_fd, &options ) != 0 ) {
        fprintf( stderr, "Failed to get current options\n" );
        ser_close();

        return -errno;
    }

    original_options = options;

    speed_t speed = B57600;
    int numerical_speed = 57600;
    if ( config->serial_speed == 9600 ) {
        speed = B9600;
        numerical_speed = 9600;
    } else if ( config->serial_speed == 19200 ) {
        speed = B19200;
        numerical_speed = 19200;
    } else if ( config->serial_speed == 38400 ) {
        speed = B38400;
        numerical_speed = 38400;
    } else if ( config->serial_speed == 115200 ) {
        speed = B115200;
        numerical_speed = 115200;
    } else if ( config->serial_speed != 57600 ) {
        fprintf( stderr, "Unsupported serial speed, using default (57600) instead\n" );
    }

    if ( BE_VERBOSE ) {
        printf( "Using serial speed %d\n", numerical_speed );
    }

    if ( cfsetispeed( &options, speed ) != 0 ) {
        fprintf( stderr, "Failed to set input speed\n" );
        ser_close();

        return -errno;
    }

    if ( cfsetospeed( &options, speed ) != 0 ) {
        fprintf( stderr, "Failed to set output speed\n" );
        ser_close();

        return -errno;
    }

    options.c_cflag |= ( CLOCAL | CREAD );

    options.c_cflag &= ~PARENB;
    options.c_cflag &= ~CSTOPB;
    options.c_cflag &= ~CSIZE;
    options.c_cflag |= CS8;

    options.c_cflag |= CRTSCTS;
    //options.c_cflag &= ~CRTSCTS;

    options.c_lflag &= ~( ICANON | ECHO | ECHOE | ISIG );

    // reset all input flags to avoid any input modifications
    options.c_iflag = 0;

    // redundant, but we explicitly don't want software flow control
    options.c_iflag &= ~(IXON | IXOFF | IXANY);

    // clear all post processing options
    options.c_oflag = 0;

    // set read behavior: we know exactly when bytes are coming see
    // read should block until atleast one byte is available.
    options.c_cc[VTIME] = 0;
    options.c_cc[VMIN] = 1;

    if ( tcsetattr( serial_fd, TCSAFLUSH, &options ) != 0 ) {
        fprintf( stderr, "Failed to set new parameters\n" );
        ser_close();

        return -errno;
    }

    printf( "Serial port %s initialized\n", tty_name );

    return 0;    
}

int pc_init_port( const struct apccomm_transfer_config *config )
{
    struct sigaction act;

    if ( ! config ) {
        return -EINVAL;
    }

    memset( &act, 0, sizeof( act ) );
    act.sa_handler = &sighandler;
    sigemptyset( &act.sa_mask );

    if ( config->mode == APCCOMM_PARALLEL ) {
        if ( initParPort( config ) != 0 ) {
            fprintf( stderr, "Failed to init parallel port\n" );
            return -EINVAL;
        }

        if ( sigaction( SIGINT, &act, NULL ) == -1) {
            perror("sigaction");
            pc_cleanup_par();;
            return -EINVAL;
        }

        inblock_size = TRANSFERBLOCKSIZE;
        outblock_size = TRANSFERBLOCKSIZE/2;
        inblock_base = 0;
        outblock_base = 0;

        pc_sync = synchro;
        pc_transferblock = transferblock;
        pc_cleanup = pc_cleanup_par;
    } else if ( config->mode == APCCOMM_SERIAL ) {
        int res = ser_init( config );

        if ( res != 0 ) {
            fprintf( stderr, "Failed to init serial port\n" );
            return -EINVAL;
        }

        if ( sigaction( SIGINT, &act, NULL ) == -1) {
            perror("sigaction");
            ser_close();
            return -EINVAL;
        }

        inblock_size = TRANSFERBLOCKSIZE;
        outblock_size = TRANSFERBLOCKSIZE;
        inblock_base = 2;
        outblock_base = 2;

        pc_sync = ser_sync;
        pc_transferblock = ser_transfer_block;
        pc_cleanup = ser_close;
    } else {
        fprintf( stderr, "Invalid transfer mode %d\n", config->mode );
        return -EINVAL;
    }

    return 0;
}
