/*
 *  tavvva_ppdrv - ioctl based parallel port driving library
 *                 (no direct I/O access!!!)
 *
 *  Copyright (C) 2013 Jaromir Capik <tavvva@email.cz>
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "tavvva_ppdrv.h"

#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <linux/ppdev.h>

#include <stdio.h>
#include <unistd.h>

int fd;
int tpp_active_ = 0;
unsigned char old_data;

/* file ... /dev/parportX */
int tpp_init (char *file)
{
  fd = open (file, O_RDWR);

  if ( fd == -1 ) {
    fprintf (stderr, "Can't open %s\n", file);
    return 1;
  }

  if ( ioctl (fd, PPCLAIM) ) {
    fprintf (stderr, "PPCLAIM failed\n");
    close (fd);
    return 1;
  }

  tpp_active_ = 1;

  return 0;
}


int tpp_exit (void)
{
  if ( ioctl (fd, PPRELEASE) ) {
    fprintf (stderr, "PPRELEASE failed\n");
    close(fd);
    return 1;
  }

  close(fd);

  tpp_active_ = 0;

  return 0;
}


int tpp_active (void)
{
  return tpp_active_;
}


int tpp_read_control (unsigned char *ctrl)
{
  if ( ioctl (fd, PPRCONTROL, ctrl) ) {
    fprintf (stderr, "PPRCONTROL failed\n");
    return 1;
  }

  return 0;
}


int tpp_write_control (unsigned char ctrl)
{
  if ( ioctl (fd, PPWCONTROL, &ctrl) ) {
    fprintf (stderr, "PPWCONTROL failed\n");
    return 1;
  }

  return 0;
}


int tpp_frobs_control (unsigned char ctrl, unsigned char mask)
{
  static struct ppdev_frob_struct frb;

  frb.mask = mask;
  frb.val = ctrl;

  if ( ioctl (fd, PPFCONTROL, &frb) ) {
    fprintf (stderr, "PPFCONTROL failed\n");
    return 1;
  }

  return 0;
}


int tpp_read_status (unsigned char *status)
{
  if ( ioctl (fd, PPRSTATUS, status) ) {
    fprintf (stderr, "PPRSTATUS failed\n");
    return 1;
  }

  return 0;
}


/* 0 - output, 1 - input */
int tpp_set_data_direction (int direction)
{
  if ( ioctl (fd, PPDATADIR, &direction) ) {
    fprintf (stderr, "PPDATADIR failed\n");
    return 1;
  }

  return 0;
}


int tpp_write_data (unsigned char data)
{
  if ( ioctl (fd, PPWDATA, &data) ) {
    fprintf (stderr, "PPWDATA failed\n");
    return 1;
  }

  old_data = data;

  return 0;
}


unsigned char tpp_data_loopback (void)
{
  return old_data;
}


int tpp_read_data (unsigned char *data)
{
  if ( ioctl (fd, PPRDATA, data) ) {
    fprintf (stderr, "PPRDATA failed\n");
    return 1;
  }

  return 0;
}


int tpp_write_pin (int pin, int val)
{
  switch (pin) {
    /* strobe */
    case 1: tpp_frobs_control ( pin1_to_fctrl(val), PIN1_MASK ); break;

    /* data */
    case 2:
    case 3:
    case 4:
    case 5:
    case 6:
    case 7:
    case 8:
    case 9: pin_to_data ( pin, val, old_data ); tpp_write_data ( old_data ); break;

    /* line feed */
    case 14: tpp_frobs_control ( pin14_to_fctrl(val), PIN14_MASK ); break;

    /* init */
    case 16: tpp_frobs_control ( pin16_to_fctrl(val), PIN16_MASK ); break;

    /* select */
    case 17: tpp_frobs_control ( pin17_to_fctrl(val), PIN17_MASK ); break;

    default:
      fprintf (stderr, "Invalid output pin number!");
      return 1;
  }

  return 0;
}


int tpp_read_pin (int pin, int *val)
{
  unsigned char tmp;

  switch (pin) {
    /* data */
    case 2:
    case 3:
    case 4:
    case 5:
    case 6:
    case 7:
    case 8:
    case 9: tpp_read_data (&tmp); *val = data_to_pin(pin, tmp); break;

    /* ack */
    case 10: tpp_read_status (&tmp); *val = stat_to_pin10(tmp); break;

    /* busy */
    case 11: tpp_read_status (&tmp); *val = stat_to_pin11(tmp); break;

    /* paper out */
    case 12: tpp_read_status (&tmp); *val = stat_to_pin12(tmp); break;

    /* select */
    case 13: tpp_read_status (&tmp); *val = stat_to_pin13(tmp); break;

    /* error */
    case 15: tpp_read_status (&tmp); *val = stat_to_pin15(tmp); break;

    default:
      fprintf (stderr, "Invalid input pin number!");
      return 1;
  }

  return 0;
}
